# Copyright 2022-2024 ETSI OSG/SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from typing import Callable, Optional
import asyncio
import grpc
import logging
from common.proto.dlt_gateway_pb2 import DltRecordEvent, DltRecordSubscription
from common.tools.grpc.Tools import grpc_message_to_json_string
from dlt.connector.client.DltGatewayClient import DltGatewayClient

LOGGER = logging.getLogger(__name__)
LOGGER.setLevel(logging.DEBUG)

class DltEventsCollector:
    def __init__(
        self, dltgateway_client: DltGatewayClient,
        log_events_received: bool = False,
        event_handler: Optional[Callable[[DltRecordEvent], Optional[DltRecordEvent]]] = None,
    ) -> None:
        self._dltgateway_client = dltgateway_client
        self._log_events_received = log_events_received
        self._event_handler = event_handler
        self._events_queue = asyncio.Queue()
        self._terminate = asyncio.Event()
        self._dltgateway_stream = None

    async def run(self) -> None:
        event_handler = self._event_handler
        if event_handler is None:
            event_handler = lambda e: e
        self._terminate.clear()
        while not self._terminate.is_set():
            try:
                subscription = DltRecordSubscription()  # by default subscribe to all
                self._dltgateway_stream = await self._dltgateway_client.SubscribeToDlt(subscription)
                async for event in self._dltgateway_stream:
                    if self._log_events_received:
                        LOGGER.info('[_collect] event: {:s}'.format(grpc_message_to_json_string(event)))
                    event = event_handler(event)
                    if event is None:
                        continue
                    if not isinstance(event, DltRecordEvent):
                        raise Exception('Unsupported return type: {:s}'.format(str(event)))
                    await self._events_queue.put(event)
            except grpc.RpcError as e:
                if e.code() == grpc.StatusCode.UNAVAILABLE:  # pylint: disable=no-member
                    await asyncio.sleep(0.5)
                    continue
                elif e.code() == grpc.StatusCode.CANCELLED:  # pylint: disable=no-member
                    break
                else:
                    raise  # pragma: no cover

    async def get_event(self, block: bool = True, timeout: float = 0.1):
        try:
            return await asyncio.wait_for(self._events_queue.get(), timeout)
        except asyncio.TimeoutError:
            return None

    async def get_events(self, block: bool = True, timeout: float = 0.1, count: int = None):
        events = []
        if count is None:
            while True:
                event = await self.get_event(block=block, timeout=timeout)
                if event is None:
                    break
                events.append(event)
        else:
            for _ in range(count):
                event = await self.get_event(block=block, timeout=timeout)
                if event is None:
                    continue
                events.append(event)
        return sorted(events, key=lambda e: e.event.timestamp.timestamp)

    async def stop(self):
        self._terminate.set()
        if self._dltgateway_stream is not None:
            await self._dltgateway_stream.cancel()

