# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import copy, grpc, pytest
from common.Constants import DEFAULT_CONTEXT_UUID, DEFAULT_TOPOLOGY_UUID
from common.proto.context_pb2 import (
    Context, ContextId, Device, DeviceDriverEnum, DeviceId, DeviceOperationalStatusEnum, Empty, Topology, TopologyId)
from context.client.ContextClient import ContextClient
#from context.client.EventsCollector import EventsCollector
from .Objects import CONTEXT, CONTEXT_ID, DEVICE_R1, DEVICE_R1_ID, DEVICE_R1_UUID, TOPOLOGY, TOPOLOGY_ID

def grpc_device(context_client_grpc : ContextClient) -> None:

    # ----- Initialize the EventsCollector -----------------------------------------------------------------------------
    #events_collector = EventsCollector(
    #    context_client_grpc, log_events_received=True,
    #    activate_context_collector = False, activate_topology_collector = False, activate_device_collector = True,
    #    activate_link_collector = False, activate_service_collector = False, activate_slice_collector = False,
    #    activate_connection_collector = False)
    #events_collector.start()

    # ----- Prepare dependencies for the test and capture related events -----------------------------------------------
    response = context_client_grpc.SetContext(Context(**CONTEXT))
    assert response.context_uuid.uuid == DEFAULT_CONTEXT_UUID

    response = context_client_grpc.SetTopology(Topology(**TOPOLOGY))
    assert response.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert response.topology_uuid.uuid == DEFAULT_TOPOLOGY_UUID

    #events = events_collector.get_events(block=True, count=2)
    #assert isinstance(events[0], ContextEvent)
    #assert events[0].event.event_type == EventTypeEnum.EVENTTYPE_CREATE
    #assert events[0].context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    #assert isinstance(events[1], TopologyEvent)
    #assert events[1].event.event_type == EventTypeEnum.EVENTTYPE_CREATE
    #assert events[1].topology_id.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    #assert events[1].topology_id.topology_uuid.uuid == DEFAULT_TOPOLOGY_UUID

    # ----- Get when the object does not exist -------------------------------------------------------------------------
    with pytest.raises(grpc.RpcError) as e:
        context_client_grpc.GetDevice(DeviceId(**DEVICE_R1_ID))
    assert e.value.code() == grpc.StatusCode.NOT_FOUND
    assert e.value.details() == 'Device({:s}) not found'.format(DEVICE_R1_UUID)

    # ----- List when the object does not exist ------------------------------------------------------------------------
    response = context_client_grpc.ListDeviceIds(Empty())
    assert len(response.device_ids) == 0

    response = context_client_grpc.ListDevices(Empty())
    assert len(response.devices) == 0

    # ----- Create the object ------------------------------------------------------------------------------------------
    with pytest.raises(grpc.RpcError) as e:
        WRONG_DEVICE = copy.deepcopy(DEVICE_R1)
        WRONG_DEVICE_UUID = '3f03c76d-31fb-47f5-9c1d-bc6b6bfa2d08'
        WRONG_DEVICE['device_endpoints'][0]['endpoint_id']['device_id']['device_uuid']['uuid'] = WRONG_DEVICE_UUID
        context_client_grpc.SetDevice(Device(**WRONG_DEVICE))
    assert e.value.code() == grpc.StatusCode.INVALID_ARGUMENT
    msg = 'request.device_endpoints[0].device_id.device_uuid.uuid({}) is invalid; '\
          'should be == request.device_id.device_uuid.uuid({})'.format(WRONG_DEVICE_UUID, DEVICE_R1_UUID)
    assert e.value.details() == msg

    response = context_client_grpc.SetDevice(Device(**DEVICE_R1))
    assert response.device_uuid.uuid == DEVICE_R1_UUID

    # ----- Check create event -----------------------------------------------------------------------------------------
    # event = events_collector.get_event(block=True)
    # assert isinstance(event, DeviceEvent)
    # assert event.event.event_type == EventTypeEnum.EVENTTYPE_CREATE
    # assert event.device_id.device_uuid.uuid == DEVICE_R1_UUID

    # ----- Get when the object exists ---------------------------------------------------------------------------------
    response = context_client_grpc.GetDevice(DeviceId(**DEVICE_R1_ID))
    assert response.device_id.device_uuid.uuid == DEVICE_R1_UUID
    assert response.name == ''
    assert response.device_type == 'packet-router'
    assert len(response.device_config.config_rules) == 3
    assert response.device_operational_status == DeviceOperationalStatusEnum.DEVICEOPERATIONALSTATUS_DISABLED
    assert len(response.device_drivers) == 1
    assert DeviceDriverEnum.DEVICEDRIVER_OPENCONFIG in response.device_drivers
    assert len(response.device_endpoints) == 3

    # ----- List when the object exists --------------------------------------------------------------------------------
    response = context_client_grpc.ListDeviceIds(Empty())
    assert len(response.device_ids) == 1
    assert response.device_ids[0].device_uuid.uuid == DEVICE_R1_UUID

    response = context_client_grpc.ListDevices(Empty())
    assert len(response.devices) == 1
    assert response.devices[0].device_id.device_uuid.uuid == DEVICE_R1_UUID
    assert response.devices[0].name == ''
    assert response.devices[0].device_type == 'packet-router'
    assert len(response.devices[0].device_config.config_rules) == 3
    assert response.devices[0].device_operational_status == DeviceOperationalStatusEnum.DEVICEOPERATIONALSTATUS_DISABLED
    assert len(response.devices[0].device_drivers) == 1
    assert DeviceDriverEnum.DEVICEDRIVER_OPENCONFIG in response.devices[0].device_drivers
    assert len(response.devices[0].device_endpoints) == 3

    # ----- Update the object ------------------------------------------------------------------------------------------
    new_device_name = 'r1'
    new_device_driver = DeviceDriverEnum.DEVICEDRIVER_UNDEFINED
    DEVICE_UPDATED = copy.deepcopy(DEVICE_R1)
    DEVICE_UPDATED['name'] = new_device_name
    DEVICE_UPDATED['device_operational_status'] = DeviceOperationalStatusEnum.DEVICEOPERATIONALSTATUS_ENABLED
    DEVICE_UPDATED['device_drivers'].append(new_device_driver)
    response = context_client_grpc.SetDevice(Device(**DEVICE_UPDATED))
    assert response.device_uuid.uuid == DEVICE_R1_UUID

    # ----- Check update event -----------------------------------------------------------------------------------------
    # event = events_collector.get_event(block=True)
    # assert isinstance(event, DeviceEvent)
    # assert event.event.event_type == EventTypeEnum.EVENTTYPE_UPDATE
    # assert event.device_id.device_uuid.uuid == DEVICE_R1_UUID

    # ----- Get when the object is modified ----------------------------------------------------------------------------
    response = context_client_grpc.GetDevice(DeviceId(**DEVICE_R1_ID))
    assert response.device_id.device_uuid.uuid == DEVICE_R1_UUID
    assert response.name == new_device_name
    assert response.device_type == 'packet-router'
    assert len(response.device_config.config_rules) == 3
    assert response.device_operational_status == DeviceOperationalStatusEnum.DEVICEOPERATIONALSTATUS_ENABLED
    assert len(response.device_drivers) == 2
    assert DeviceDriverEnum.DEVICEDRIVER_UNDEFINED in response.device_drivers
    assert DeviceDriverEnum.DEVICEDRIVER_OPENCONFIG in response.device_drivers
    assert len(response.device_endpoints) == 3

    # ----- List when the object is modified ---------------------------------------------------------------------------
    response = context_client_grpc.ListDeviceIds(Empty())
    assert len(response.device_ids) == 1
    assert response.device_ids[0].device_uuid.uuid == DEVICE_R1_UUID

    response = context_client_grpc.ListDevices(Empty())
    assert len(response.devices) == 1
    assert response.devices[0].device_id.device_uuid.uuid == DEVICE_R1_UUID
    assert response.devices[0].name == new_device_name
    assert response.devices[0].device_type == 'packet-router'
    assert len(response.devices[0].device_config.config_rules) == 3
    assert response.devices[0].device_operational_status == DeviceOperationalStatusEnum.DEVICEOPERATIONALSTATUS_ENABLED
    assert len(response.devices[0].device_drivers) == 2
    assert DeviceDriverEnum.DEVICEDRIVER_UNDEFINED in response.devices[0].device_drivers
    assert DeviceDriverEnum.DEVICEDRIVER_OPENCONFIG in response.devices[0].device_drivers
    assert len(response.devices[0].device_endpoints) == 3

    # ----- Create object relation -------------------------------------------------------------------------------------
    TOPOLOGY_WITH_DEVICE = copy.deepcopy(TOPOLOGY)
    TOPOLOGY_WITH_DEVICE['device_ids'].append(DEVICE_R1_ID)
    response = context_client_grpc.SetTopology(Topology(**TOPOLOGY_WITH_DEVICE))
    assert response.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert response.topology_uuid.uuid == DEFAULT_TOPOLOGY_UUID

    # ----- Check update event -----------------------------------------------------------------------------------------
    # event = events_collector.get_event(block=True)
    # assert isinstance(event, TopologyEvent)
    # assert event.event.event_type == EventTypeEnum.EVENTTYPE_UPDATE
    # assert response.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    # assert response.topology_uuid.uuid == DEFAULT_TOPOLOGY_UUID

    # ----- Check relation was created ---------------------------------------------------------------------------------
    response = context_client_grpc.GetTopology(TopologyId(**TOPOLOGY_ID))
    assert response.topology_id.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert response.topology_id.topology_uuid.uuid == DEFAULT_TOPOLOGY_UUID
    assert len(response.device_ids) == 1
    assert response.device_ids[0].device_uuid.uuid == DEVICE_R1_UUID
    assert len(response.link_ids) == 0

    # ----- Remove the object ------------------------------------------------------------------------------------------
    context_client_grpc.RemoveDevice(DeviceId(**DEVICE_R1_ID))
    context_client_grpc.RemoveTopology(TopologyId(**TOPOLOGY_ID))
    context_client_grpc.RemoveContext(ContextId(**CONTEXT_ID))

    # ----- Check remove event -----------------------------------------------------------------------------------------
    # events = events_collector.get_events(block=True, count=3)

    # assert isinstance(events[0], DeviceEvent)
    # assert events[0].event.event_type == EventTypeEnum.EVENTTYPE_REMOVE
    # assert events[0].device_id.device_uuid.uuid == DEVICE_R1_UUID

    # assert isinstance(events[1], TopologyEvent)
    # assert events[1].event.event_type == EventTypeEnum.EVENTTYPE_REMOVE
    # assert events[1].topology_id.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    # assert events[1].topology_id.topology_uuid.uuid == DEFAULT_TOPOLOGY_UUID

    # assert isinstance(events[2], ContextEvent)
    # assert events[2].event.event_type == EventTypeEnum.EVENTTYPE_REMOVE
    # assert events[2].context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID

    # ----- Stop the EventsCollector -----------------------------------------------------------------------------------
    #events_collector.stop()
