# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import copy, grpc, pytest, uuid
from common.Constants import DEFAULT_CONTEXT_UUID
from common.proto.context_pb2 import Context, ContextId, Empty
from common.tools.object_factory.Context import json_context_id
from common.tools.object_factory.Service import json_service_id
from common.tools.object_factory.Slice import json_slice_id
from common.tools.object_factory.Topology import json_topology_id
from context.client.ContextClient import ContextClient
#from context.client.EventsCollector import EventsCollector
from .Objects import CONTEXT, CONTEXT_ID

def grpc_context(context_client_grpc : ContextClient) -> None:

    # ----- Initialize the EventsCollector -----------------------------------------------------------------------------
    #events_collector = EventsCollector(
    #    context_client_grpc, log_events_received=True,
    #    activate_context_collector = True, activate_topology_collector = False, activate_device_collector = False,
    #    activate_link_collector = False, activate_service_collector = False, activate_slice_collector = False,
    #    activate_connection_collector = False)
    #events_collector.start()

    # ----- Get when the object does not exist -------------------------------------------------------------------------
    with pytest.raises(grpc.RpcError) as e:
        context_client_grpc.GetContext(ContextId(**CONTEXT_ID))
    assert e.value.code() == grpc.StatusCode.NOT_FOUND
    assert e.value.details() == 'Context({:s}) not found'.format(DEFAULT_CONTEXT_UUID)

    # ----- List when the object does not exist ------------------------------------------------------------------------
    response = context_client_grpc.ListContextIds(Empty())
    assert len(response.context_ids) == 0

    response = context_client_grpc.ListContexts(Empty())
    assert len(response.contexts) == 0

    # ----- Create the object ------------------------------------------------------------------------------------------
    response = context_client_grpc.SetContext(Context(**CONTEXT))
    assert response.context_uuid.uuid == DEFAULT_CONTEXT_UUID

    wrong_context_uuid = str(uuid.uuid4())
    wrong_context_id = json_context_id(wrong_context_uuid)
    with pytest.raises(grpc.RpcError) as e:
        WRONG_CONTEXT = copy.deepcopy(CONTEXT)
        WRONG_CONTEXT['topology_ids'].append(json_topology_id(str(uuid.uuid4()), context_id=wrong_context_id))
        context_client_grpc.SetContext(Context(**WRONG_CONTEXT))
    assert e.value.code() == grpc.StatusCode.INVALID_ARGUMENT
    msg = 'request.topology_ids[0].context_id.context_uuid.uuid({}) is invalid; '\
          'should be == request.context_id.context_uuid.uuid({})'.format(wrong_context_uuid, DEFAULT_CONTEXT_UUID)
    assert e.value.details() == msg

    with pytest.raises(grpc.RpcError) as e:
        WRONG_CONTEXT = copy.deepcopy(CONTEXT)
        WRONG_CONTEXT['service_ids'].append(json_service_id(str(uuid.uuid4()), context_id=wrong_context_id))
        context_client_grpc.SetContext(Context(**WRONG_CONTEXT))
    assert e.value.code() == grpc.StatusCode.INVALID_ARGUMENT
    msg = 'request.service_ids[0].context_id.context_uuid.uuid({}) is invalid; '\
          'should be == request.context_id.context_uuid.uuid({})'.format(wrong_context_uuid, DEFAULT_CONTEXT_UUID)
    assert e.value.details() == msg

    with pytest.raises(grpc.RpcError) as e:
        WRONG_CONTEXT = copy.deepcopy(CONTEXT)
        WRONG_CONTEXT['slice_ids'].append(json_slice_id(str(uuid.uuid4()), context_id=wrong_context_id))
        context_client_grpc.SetContext(Context(**WRONG_CONTEXT))
    assert e.value.code() == grpc.StatusCode.INVALID_ARGUMENT
    msg = 'request.slice_ids[0].context_id.context_uuid.uuid({}) is invalid; '\
          'should be == request.context_id.context_uuid.uuid({})'.format(wrong_context_uuid, DEFAULT_CONTEXT_UUID)
    assert e.value.details() == msg

    # ----- Check create event -----------------------------------------------------------------------------------------
    #event = events_collector.get_event(block=True, timeout=10.0)
    #assert isinstance(event, ContextEvent)
    #assert event.event.event_type == EventTypeEnum.EVENTTYPE_CREATE
    #assert event.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID

    # ----- Get when the object exists ---------------------------------------------------------------------------------
    response = context_client_grpc.GetContext(ContextId(**CONTEXT_ID))
    assert response.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert response.name == ''
    assert len(response.topology_ids) == 0
    assert len(response.service_ids) == 0
    assert len(response.slice_ids) == 0

    # ----- List when the object exists --------------------------------------------------------------------------------
    response = context_client_grpc.ListContextIds(Empty())
    assert len(response.context_ids) == 1
    assert response.context_ids[0].context_uuid.uuid == DEFAULT_CONTEXT_UUID

    response = context_client_grpc.ListContexts(Empty())
    assert len(response.contexts) == 1
    assert response.contexts[0].context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert response.contexts[0].name == ''
    assert len(response.contexts[0].topology_ids) == 0
    assert len(response.contexts[0].service_ids) == 0
    assert len(response.contexts[0].slice_ids) == 0

    # ----- Update the object ------------------------------------------------------------------------------------------
    new_context_name = 'new'
    CONTEXT_WITH_NAME = copy.deepcopy(CONTEXT)
    CONTEXT_WITH_NAME['name'] = new_context_name
    response = context_client_grpc.SetContext(Context(**CONTEXT_WITH_NAME))
    assert response.context_uuid.uuid == DEFAULT_CONTEXT_UUID

    # ----- Check update event -----------------------------------------------------------------------------------------
    #event = events_collector.get_event(block=True, timeout=10.0)
    #assert isinstance(event, ContextEvent)
    #assert event.event.event_type == EventTypeEnum.EVENTTYPE_UPDATE
    #assert event.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID

    # ----- Get when the object is modified ----------------------------------------------------------------------------
    response = context_client_grpc.GetContext(ContextId(**CONTEXT_ID))
    assert response.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert response.name == new_context_name
    assert len(response.topology_ids) == 0
    assert len(response.service_ids) == 0
    assert len(response.slice_ids) == 0

    # ----- List when the object is modified ---------------------------------------------------------------------------
    response = context_client_grpc.ListContextIds(Empty())
    assert len(response.context_ids) == 1
    assert response.context_ids[0].context_uuid.uuid == DEFAULT_CONTEXT_UUID

    response = context_client_grpc.ListContexts(Empty())
    assert len(response.contexts) == 1
    assert response.contexts[0].context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert response.contexts[0].name == new_context_name
    assert len(response.contexts[0].topology_ids) == 0
    assert len(response.contexts[0].service_ids) == 0
    assert len(response.contexts[0].slice_ids) == 0

    # ----- Remove the object ------------------------------------------------------------------------------------------
    context_client_grpc.RemoveContext(ContextId(**CONTEXT_ID))

    # ----- Check remove event -----------------------------------------------------------------------------------------
    #event = events_collector.get_event(block=True, timeout=10.0)
    #assert isinstance(event, ContextEvent)
    #assert event.event.event_type == EventTypeEnum.EVENTTYPE_REMOVE
    #assert event.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID

    # ----- List after deleting the object -----------------------------------------------------------------------------
    response = context_client_grpc.ListContextIds(Empty())
    assert len(response.context_ids) == 0

    response = context_client_grpc.ListContexts(Empty())
    assert len(response.contexts) == 0

    # ----- Stop the EventsCollector -----------------------------------------------------------------------------------
    #events_collector.stop()
