# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from common.Constants import DEFAULT_CONTEXT_UUID, DEFAULT_TOPOLOGY_UUID
from common.proto.kpi_sample_types_pb2 import KpiSampleType
from common.tools.object_factory.ConfigRule import json_config_rule_set
from common.tools.object_factory.Connection import json_connection, json_connection_id
from common.tools.object_factory.Constraint import json_constraint_custom
from common.tools.object_factory.Context import json_context, json_context_id
from common.tools.object_factory.Device import json_device_id, json_device_packetrouter_disabled
from common.tools.object_factory.EndPoint import json_endpoint, json_endpoint_id
from common.tools.object_factory.Link import json_link, json_link_id
from common.tools.object_factory.Service import json_service_id, json_service_l3nm_planned
from common.tools.object_factory.Topology import json_topology, json_topology_id
from common.tools.object_factory.PolicyRule import json_policy_rule, json_policy_rule_id


# ----- Context --------------------------------------------------------------------------------------------------------
CONTEXT_ID = json_context_id(DEFAULT_CONTEXT_UUID)
CONTEXT    = json_context(DEFAULT_CONTEXT_UUID)


# ----- Topology -------------------------------------------------------------------------------------------------------
TOPOLOGY_ID = json_topology_id(DEFAULT_TOPOLOGY_UUID, context_id=CONTEXT_ID)
TOPOLOGY    = json_topology(DEFAULT_TOPOLOGY_UUID, context_id=CONTEXT_ID)


# ----- KPI Sample Types -----------------------------------------------------------------------------------------------
PACKET_PORT_SAMPLE_TYPES = [
    KpiSampleType.KPISAMPLETYPE_PACKETS_TRANSMITTED,
    KpiSampleType.KPISAMPLETYPE_PACKETS_RECEIVED,
    KpiSampleType.KPISAMPLETYPE_BYTES_TRANSMITTED,
    KpiSampleType.KPISAMPLETYPE_BYTES_RECEIVED,
]


# ----- Device ---------------------------------------------------------------------------------------------------------
EP1 = '5610e2c0-8abe-4127-80d0-7c68aff1c19e'
EP2 = '7eb80584-2587-4e71-b10c-f3a5c48e84ab'
EP3 = '368baf47-0540-4ab4-add8-a19b5167162c'
EP100 = '6a923121-36e1-4b5e-8cd6-90aceca9b5cf'


DEVICE_R1_UUID  = 'fe83a200-6ded-47b4-b156-3bb3556a10d6'
DEVICE_R1_ID    = json_device_id(DEVICE_R1_UUID)
DEVICE_R1_EPS   = [
    json_endpoint(DEVICE_R1_ID, EP2,   '10G', topology_id=TOPOLOGY_ID, kpi_sample_types=PACKET_PORT_SAMPLE_TYPES),
    json_endpoint(DEVICE_R1_ID, EP3,   '10G', topology_id=TOPOLOGY_ID, kpi_sample_types=PACKET_PORT_SAMPLE_TYPES),
    json_endpoint(DEVICE_R1_ID, EP100, '10G', topology_id=TOPOLOGY_ID, kpi_sample_types=PACKET_PORT_SAMPLE_TYPES),
]
DEVICE_R1_RULES = [
    json_config_rule_set('dev/rsrc1/value', 'value1'),
    json_config_rule_set('dev/rsrc2/value', 'value2'),
    json_config_rule_set('dev/rsrc3/value', 'value3'),
]
DEVICE_R1       = json_device_packetrouter_disabled(
    DEVICE_R1_UUID, endpoints=DEVICE_R1_EPS, config_rules=DEVICE_R1_RULES)


DEVICE_R2_UUID  = '2fd2be23-5b20-414c-b1ea-2f16ae6eb425'
DEVICE_R2_ID    = json_device_id(DEVICE_R2_UUID)
DEVICE_R2_EPS   = [
    json_endpoint(DEVICE_R2_ID, EP1,   '10G', topology_id=TOPOLOGY_ID, kpi_sample_types=PACKET_PORT_SAMPLE_TYPES),
    json_endpoint(DEVICE_R2_ID, EP3,   '10G', topology_id=TOPOLOGY_ID, kpi_sample_types=PACKET_PORT_SAMPLE_TYPES),
    json_endpoint(DEVICE_R2_ID, EP100, '10G', topology_id=TOPOLOGY_ID, kpi_sample_types=PACKET_PORT_SAMPLE_TYPES),
]
DEVICE_R2_RULES = [
    json_config_rule_set('dev/rsrc1/value', 'value4'),
    json_config_rule_set('dev/rsrc2/value', 'value5'),
    json_config_rule_set('dev/rsrc3/value', 'value6'),
]
DEVICE_R2       = json_device_packetrouter_disabled(
    DEVICE_R2_UUID, endpoints=DEVICE_R2_EPS, config_rules=DEVICE_R2_RULES)


DEVICE_R3_UUID  = '3e71a251-2218-42c5-b4b8-de7760c0d9b3'
DEVICE_R3_ID    = json_device_id(DEVICE_R3_UUID)
DEVICE_R3_EPS   = [
    json_endpoint(DEVICE_R3_ID, EP2,   '10G', topology_id=TOPOLOGY_ID, kpi_sample_types=PACKET_PORT_SAMPLE_TYPES),
    json_endpoint(DEVICE_R3_ID, EP3,   '10G', topology_id=TOPOLOGY_ID, kpi_sample_types=PACKET_PORT_SAMPLE_TYPES),
    json_endpoint(DEVICE_R3_ID, EP100, '10G', topology_id=TOPOLOGY_ID, kpi_sample_types=PACKET_PORT_SAMPLE_TYPES),
]
DEVICE_R3_RULES = [
    json_config_rule_set('dev/rsrc1/value', 'value4'),
    json_config_rule_set('dev/rsrc2/value', 'value5'),
    json_config_rule_set('dev/rsrc3/value', 'value6'),
]
DEVICE_R3       = json_device_packetrouter_disabled(
    DEVICE_R3_UUID, endpoints=DEVICE_R3_EPS, config_rules=DEVICE_R3_RULES)


# ----- Link -----------------------------------------------------------------------------------------------------------
LINK_R1_R2_UUID  = 'c8f92eec-340e-4d31-8d7e-7074927dc889'
LINK_R1_R2_ID    = json_link_id(LINK_R1_R2_UUID)
LINK_R1_R2_EPIDS = [
    json_endpoint_id(DEVICE_R1_ID, EP2, topology_id=TOPOLOGY_ID),
    json_endpoint_id(DEVICE_R2_ID, EP1, topology_id=TOPOLOGY_ID),
]
LINK_R1_R2       = json_link(LINK_R1_R2_UUID, LINK_R1_R2_EPIDS)


LINK_R2_R3_UUID  = 'f9e3539a-d8f9-4737-b4b4-cacf7f90aa5d'
LINK_R2_R3_ID    = json_link_id(LINK_R2_R3_UUID)
LINK_R2_R3_EPIDS = [
    json_endpoint_id(DEVICE_R2_ID, EP3, topology_id=TOPOLOGY_ID),
    json_endpoint_id(DEVICE_R3_ID, EP2, topology_id=TOPOLOGY_ID),
]
LINK_R2_R3       = json_link(LINK_R2_R3_UUID, LINK_R2_R3_EPIDS)


LINK_R1_R3_UUID  = '1f1a988c-47a9-41b2-afd9-ebd6d434a0b4'
LINK_R1_R3_ID    = json_link_id(LINK_R1_R3_UUID)
LINK_R1_R3_EPIDS = [
    json_endpoint_id(DEVICE_R1_ID, EP3, topology_id=TOPOLOGY_ID),
    json_endpoint_id(DEVICE_R3_ID, EP1, topology_id=TOPOLOGY_ID),
]
LINK_R1_R3       = json_link(LINK_R1_R3_UUID, LINK_R1_R3_EPIDS)


# ----- Service --------------------------------------------------------------------------------------------------------
SERVICE_R1_R2_UUID  = 'f0432e7b-bb83-4880-9c5d-008c4925ce7d'
SERVICE_R1_R2_ID    = json_service_id(SERVICE_R1_R2_UUID, context_id=CONTEXT_ID)
SERVICE_R1_R2_EPIDS = [
    json_endpoint_id(DEVICE_R1_ID, EP100, topology_id=TOPOLOGY_ID),
    json_endpoint_id(DEVICE_R2_ID, EP100, topology_id=TOPOLOGY_ID),
]
SERVICE_R1_R2_CONST = [
    json_constraint_custom('latency[ms]', '15.2'),
    json_constraint_custom('jitter[us]',  '1.2'),
]
SERVICE_R1_R2_RULES = [
    json_config_rule_set('svc/rsrc1/value', 'value7'),
    json_config_rule_set('svc/rsrc2/value', 'value8'),
    json_config_rule_set('svc/rsrc3/value', 'value9'),
]
SERVICE_R1_R2       = json_service_l3nm_planned(
    SERVICE_R1_R2_UUID, endpoint_ids=SERVICE_R1_R2_EPIDS, constraints=SERVICE_R1_R2_CONST,
    config_rules=SERVICE_R1_R2_RULES)


SERVICE_R1_R3_UUID  = 'fab21cef-542a-4948-bb4a-a0468abfa925'
SERVICE_R1_R3_ID    = json_service_id(SERVICE_R1_R3_UUID, context_id=CONTEXT_ID)
SERVICE_R1_R3_EPIDS = [
    json_endpoint_id(DEVICE_R1_ID, 'EP100', topology_id=TOPOLOGY_ID),
    json_endpoint_id(DEVICE_R3_ID, 'EP100', topology_id=TOPOLOGY_ID),
]
SERVICE_R1_R3_CONST = [
    json_constraint_custom('latency[ms]', '5.8'),
    json_constraint_custom('jitter[us]',  '0.1'),
]
SERVICE_R1_R3_RULES = [
    json_config_rule_set('svc/rsrc1/value', 'value7'),
    json_config_rule_set('svc/rsrc2/value', 'value8'),
    json_config_rule_set('svc/rsrc3/value', 'value9'),
]
SERVICE_R1_R3       = json_service_l3nm_planned(
    SERVICE_R1_R3_UUID, endpoint_ids=SERVICE_R1_R3_EPIDS, constraints=SERVICE_R1_R3_CONST,
    config_rules=SERVICE_R1_R3_RULES)


SERVICE_R2_R3_UUID  = '1f2a808f-62bb-4eaa-94fb-448ed643e61a'
SERVICE_R2_R3_ID    = json_service_id(SERVICE_R2_R3_UUID, context_id=CONTEXT_ID)
SERVICE_R2_R3_EPIDS = [
    json_endpoint_id(DEVICE_R2_ID, 'EP100', topology_id=TOPOLOGY_ID),
    json_endpoint_id(DEVICE_R3_ID, 'EP100', topology_id=TOPOLOGY_ID),
]
SERVICE_R2_R3_CONST = [
    json_constraint_custom('latency[ms]', '23.1'),
    json_constraint_custom('jitter[us]',  '3.4'),
]
SERVICE_R2_R3_RULES = [
    json_config_rule_set('svc/rsrc1/value', 'value7'),
    json_config_rule_set('svc/rsrc2/value', 'value8'),
    json_config_rule_set('svc/rsrc3/value', 'value9'),
]
SERVICE_R2_R3       = json_service_l3nm_planned(
    SERVICE_R2_R3_UUID, endpoint_ids=SERVICE_R2_R3_EPIDS, constraints=SERVICE_R2_R3_CONST,
    config_rules=SERVICE_R2_R3_RULES)


# ----- Connection -----------------------------------------------------------------------------------------------------
CONNECTION_R1_R3_UUID   = 'CON:R1/EP100-R3/EP100'
CONNECTION_R1_R3_ID     = json_connection_id(CONNECTION_R1_R3_UUID)
CONNECTION_R1_R3_EPIDS  = [
    json_endpoint_id(DEVICE_R1_ID, 'EP100', topology_id=TOPOLOGY_ID),
    json_endpoint_id(DEVICE_R1_ID, 'EP2',   topology_id=TOPOLOGY_ID),
    json_endpoint_id(DEVICE_R2_ID, 'EP1',   topology_id=TOPOLOGY_ID),
    json_endpoint_id(DEVICE_R2_ID, 'EP3',   topology_id=TOPOLOGY_ID),
    json_endpoint_id(DEVICE_R3_ID, 'EP2',   topology_id=TOPOLOGY_ID),
    json_endpoint_id(DEVICE_R3_ID, 'EP100', topology_id=TOPOLOGY_ID),
]
CONNECTION_R1_R3_SVCIDS = [SERVICE_R1_R2_ID, SERVICE_R2_R3_ID]
CONNECTION_R1_R3        = json_connection(
    CONNECTION_R1_R3_UUID, service_id=SERVICE_R1_R3_ID, path_hops_endpoint_ids=CONNECTION_R1_R3_EPIDS,
    sub_service_ids=CONNECTION_R1_R3_SVCIDS)


# ----- PolicyRule -------------------------------------------------------------------------------------------------------
POLICY_RULE_UUID = '56380225-3e40-4f74-9162-529f8dcb96a1'
POLICY_RULE_ID   = json_policy_rule_id(POLICY_RULE_UUID)
POLICY_RULE      = json_policy_rule(POLICY_RULE_UUID)
