from __future__ import annotations
from typing import TYPE_CHECKING, Dict, List
from .._engine.object._Object import _Object
from .._engine.object.Attributes import Attributes
from .._engine.object.Collection import Collection
from ._Keys import KEY_DEVICE, KEY_DEVICE_DRIVERS, KEY_DEVICE_ENDPOINTS
from .DeviceOperationalStatus import DeviceOperationalStatus, to_deviceoperationalstatus_enum
from .Endpoint import Endpoint

if TYPE_CHECKING:
    from .Context import Context
    from .Topology import Topology

VALIDATORS = {
    'device_type': lambda v: v is not None and isinstance(v, str) and (len(v) > 0),
    'device_config': lambda v: v is not None and isinstance(v, str) and (len(v) > 0),
    'device_operational_status': lambda v: v is not None and isinstance(v, DeviceOperationalStatus),
}

TRANSCODERS = {
    'device_operational_status': {
        DeviceOperationalStatus: lambda v: v.value,
        int                    : lambda v: to_deviceoperationalstatus_enum(v),
        str                    : lambda v: to_deviceoperationalstatus_enum(v),
    }
}

class Device(_Object):
    def __init__(self, device_uuid : str, parent : 'Topology'):
        super().__init__(parent, device_uuid)
        self._attributes = Attributes(parent, KEY_DEVICE, VALIDATORS, TRANSCODERS)
        self._drivers = Collection(self, KEY_DEVICE_DRIVERS)
        self._endpoints = Collection(self, KEY_DEVICE_ENDPOINTS)

    @property
    def parent(self) -> 'Topology': return self._parent

    @property
    def context(self) -> 'Context': return self.parent.context

    @property
    def context_uuid(self) -> str: return self.parent.context_uuid

    @property
    def topology(self) -> 'Topology': return self.parent

    @property
    def topology_uuid(self) -> str: return self.parent.topology_uuid

    @property
    def device_uuid(self) -> str: return self._object_uuid

    @property
    def attributes(self) -> Attributes: return self._attributes

    @property
    def drivers(self) -> Collection: return self._drivers

    @property
    def endpoints(self) -> Collection: return self._endpoints

    def endpoint(self, endpoint_uuid : str) -> Endpoint: return Endpoint(endpoint_uuid, self)

    def create(
        self, device_type : str, device_config : str, device_operational_status : DeviceOperationalStatus) -> 'Device':

        self.update(update_attributes={
            'device_type': device_type,
            'device_config': device_config,
            'device_operational_status': device_operational_status,
        })
        self.parent.devices.add(self.device_uuid)
        return self

    def update(self, update_attributes={}, remove_attributes=[]) -> 'Device':
        self.attributes.update(update_attributes=update_attributes, remove_attributes=remove_attributes)
        return self

    def delete(self) -> None:
        for endpoint_uuid in self.endpoints.get(): self.endpoint(endpoint_uuid).delete()
        self.drivers.clear()
        self.attributes.delete()
        self.parent.devices.delete(self.device_uuid)

    def dump_id(self) -> Dict:
        return {'device_uuid': {'uuid': self.device_uuid}}

    def dump_drivers(self) -> List[str]:
        return self.drivers.get()

    def dump_endpoints(self) -> List[Dict]:
        return [self.endpoint(endpoint_uuid).dump() for endpoint_uuid in self.endpoints.get()]

    def dump(self, include_drivers=True, include_endpoints=True) -> Dict:
        attributes = self.attributes.get()
        device_operational_status = attributes.get('device_operational_status', None)
        if isinstance(device_operational_status, DeviceOperationalStatus):
            device_operational_status = device_operational_status.value
        result = {
            'device_id': self.dump_id(),
            'device_type': attributes.get('device_type', None),
            'device_config': {'device_config': attributes.get('device_config', None)},
            'device_operational_status': device_operational_status,
        }
        if include_drivers: result['device_drivers'] = self.dump_drivers()
        if include_endpoints: result['endpoints'] = self.dump_endpoints()
        return result
