# Copyright 2022-2023 ETSI TeraFlowSDN - TFS OSG (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import json
from sqlalchemy import Column, DateTime, ForeignKey, String
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy.orm import relationship
from typing import Dict
from ._Base import _Base                                                

class ComponentModel(_Base):                                            #Inherited functionality from the base class _Base 
    __tablename__ = 'device_component'                                  #Name of the table in the DB associtaed with this model

    component_uuid  = Column(UUID(as_uuid=False), primary_key=True)     #Unique identifier that serves as a primary key for this table
    device_uuid     = Column(ForeignKey('device.device_uuid',ondelete='CASCADE' ), nullable=False, index=True)  #Foreign Key relationship with the field device_uuid from the Device table (CASCADE' behavior for deletion, meaning when a device is deleted, its components will also be dele)
    # component_name  = Column(String, nullable=False)                    #String field that stores the name of the component
    data            = Column(String, nullable=False)                    #String field that stores data about the component 
    created_at      = Column(DateTime, nullable=False)                  #Stores the creaton timestamp for the component
    updated_at      = Column(DateTime, nullable=False)                  #Stores the last upadted timestamp for the component

    device           = relationship('DeviceModel', back_populates='components')# lazy='selectin'#Defines a relationship between ComponentModel and DeviceModel
    #Represents a 1:n relationship where 1 device -> N components // The relationship is defined by the FK device_uuid
    def dump_id(self) -> Dict:
        return{
            'device_id'     : self.device.dump_id(),
            'component_uuid': {'uuid': self.component_uuid},
        }

    def dump(self) -> Dict:
        return {
            'component_id'  : self.dump_id(),
            'data'          : self.data,  
        }

    def dump_name(self) -> Dict:
        return {
            'component_id'  : self.dump_id(),
            'device_name'   : self.device.device_name,
            'component_name': self.name,
        }
