# Copyright 2022-2024 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import logging
from common.tools.descriptor.Loader import DescriptorLoader, check_descriptor_load_results
from common.tools.grpc.Tools import grpc_message_to_json_string
from context.client.ContextClient import ContextClient
from device.client.DeviceClient import DeviceClient
from tests.Fixtures import context_client, device_client # pylint: disable=unused-import
from tests.tools.test_tools_p4 import *

LOGGER = logging.getLogger(__name__)
LOGGER.setLevel(logging.DEBUG)

def test_initial_context(
    context_client : ContextClient # pylint: disable=redefined-outer-name
) -> None:
    # Verify the scenario has 0 service and 0 slices
    response = context_client.GetContext(ADMIN_CONTEXT_ID)
    assert len(response.service_ids) == 0
    assert len(response.slice_ids) == 0

    # Check there are no slices
    response = context_client.ListSlices(ADMIN_CONTEXT_ID)
    LOGGER.warning('Slices[{:d}] = {:s}'.format(len(response.slices), grpc_message_to_json_string(response)))
    assert len(response.slices) == 0

    # Check there is 0 service
    response = context_client.ListServices(ADMIN_CONTEXT_ID)
    LOGGER.warning('Services[{:d}] = {:s}'.format(len(response.services), grpc_message_to_json_string(response)))
    assert len(response.services) == 0

    # Check there are 3 devices
    response = context_client.ListDevices(ADMIN_CONTEXT_ID)
    LOGGER.warning('Devices[{:d}] = {:s}'.format(len(response.devices), grpc_message_to_json_string(response)))
    assert len(response.devices) == DEV_NB

def test_rules_before_insertion(
    context_client : ContextClient # pylint: disable=redefined-outer-name
) -> None:
    response = context_client.ListDevices(ADMIN_CONTEXT_ID)
    LOGGER.warning('Devices[{:d}] = {:s}'.format(len(response.devices), grpc_message_to_json_string(response)))
    assert len(response.devices) == DEV_NB

    # Verify that the following rules are in place
    desired_rules_nb = \
        CONNECTION_RULES + \
        ENDPOINT_RULES
    verify_number_of_rules(response.devices, desired_rules_nb)

def test_rules_insertion_int_batch_1(
    context_client : ContextClient, # pylint: disable=redefined-outer-name
    device_client : DeviceClient    # pylint: disable=redefined-outer-name
) -> None:
    # Load INT batch 1 rules for insertion
    descriptor_loader = DescriptorLoader(
        descriptors_file=DESC_FILE_RULES_INSERT_INT_B1, context_client=context_client, device_client=device_client
    )
    results = descriptor_loader.process()
    check_descriptor_load_results(results, descriptor_loader)

def test_rules_after_insertion_int_batch_1(
    context_client : ContextClient # pylint: disable=redefined-outer-name
) -> None:
    # State **after** inserting batch 1 of INT rules
    # Still 3 devices
    response = context_client.ListDevices(ADMIN_CONTEXT_ID)
    LOGGER.warning('Devices[{:d}] = {:s}'.format(len(response.devices), grpc_message_to_json_string(response)))
    assert len(response.devices) == DEV_NB

    # Verify that the following rules are in place
    desired_rules_nb = \
        CONNECTION_RULES + \
        ENDPOINT_RULES + \
        DATAPLANE_RULES_NB_INT_B1
    verify_number_of_rules(response.devices, desired_rules_nb)

def test_rules_insertion_int_batch_2(
    context_client : ContextClient, # pylint: disable=redefined-outer-name
    device_client : DeviceClient    # pylint: disable=redefined-outer-name
) -> None:
    # Load INT batch 2 rules for insertion
    descriptor_loader = DescriptorLoader(
        descriptors_file=DESC_FILE_RULES_INSERT_INT_B2, context_client=context_client, device_client=device_client
    )
    results = descriptor_loader.process()
    check_descriptor_load_results(results, descriptor_loader)

def test_rules_after_insertion_int_batch_2(
    context_client : ContextClient # pylint: disable=redefined-outer-name
) -> None:
    # State **after** inserting batch 2 of INT rules
    # Still 3 devices
    response = context_client.ListDevices(ADMIN_CONTEXT_ID)
    LOGGER.warning('Devices[{:d}] = {:s}'.format(len(response.devices), grpc_message_to_json_string(response)))
    assert len(response.devices) == DEV_NB

    # Verify that the following rules are in place
    desired_rules_nb = \
        CONNECTION_RULES + \
        ENDPOINT_RULES + \
        DATAPLANE_RULES_NB_INT_B1 + \
        DATAPLANE_RULES_NB_INT_B2
    verify_number_of_rules(response.devices, desired_rules_nb)

def test_rules_insertion_int_batch_3(
    context_client : ContextClient, # pylint: disable=redefined-outer-name
    device_client : DeviceClient    # pylint: disable=redefined-outer-name
) -> None:
    # Load INT batch 3 rules for insertion
    descriptor_loader = DescriptorLoader(
        descriptors_file=DESC_FILE_RULES_INSERT_INT_B3, context_client=context_client, device_client=device_client
    )
    results = descriptor_loader.process()
    check_descriptor_load_results(results, descriptor_loader)

def test_rules_after_insertion_int_batch_3(
    context_client : ContextClient # pylint: disable=redefined-outer-name
) -> None:
    # State **after** inserting batch 3 of INT rules
    # Still 3 devices
    response = context_client.ListDevices(ADMIN_CONTEXT_ID)
    LOGGER.warning('Devices[{:d}] = {:s}'.format(len(response.devices), grpc_message_to_json_string(response)))
    assert len(response.devices) == DEV_NB

    # Verify that the following rules are in place
    desired_rules_nb = \
        CONNECTION_RULES + \
        ENDPOINT_RULES + \
        DATAPLANE_RULES_NB_INT_B1 + \
        DATAPLANE_RULES_NB_INT_B2 + \
        DATAPLANE_RULES_NB_INT_B3
    verify_number_of_rules(response.devices, desired_rules_nb)

def test_rules_insertion_routing_edge_batch_4(
    context_client : ContextClient, # pylint: disable=redefined-outer-name
    device_client : DeviceClient    # pylint: disable=redefined-outer-name
) -> None:
    # Load routing edge batch 4 rules for insertion
    descriptor_loader = DescriptorLoader(
        descriptors_file=DESC_FILE_RULES_INSERT_ROUTING_EDGE, context_client=context_client, device_client=device_client
    )
    results = descriptor_loader.process()
    check_descriptor_load_results(results, descriptor_loader)

def test_rules_after_insertion_routing_edge_batch_4(
    context_client : ContextClient # pylint: disable=redefined-outer-name
) -> None:
    # State **after** inserting batch 4 of routing edge rules
    # Still 3 devices
    response = context_client.ListDevices(ADMIN_CONTEXT_ID)
    LOGGER.warning('Devices[{:d}] = {:s}'.format(len(response.devices), grpc_message_to_json_string(response)))
    assert len(response.devices) == DEV_NB

    # Verify that the following rules are in place
    desired_rules_nb = \
        CONNECTION_RULES + \
        ENDPOINT_RULES + \
        DATAPLANE_RULES_NB_INT_B1 + \
        DATAPLANE_RULES_NB_INT_B2 + \
        DATAPLANE_RULES_NB_INT_B3 + \
        DATAPLANE_RULES_NB_RT_EDGE
    verify_number_of_rules(response.devices, desired_rules_nb)

def test_rules_insertion_routing_corp_batch_5(
    context_client : ContextClient, # pylint: disable=redefined-outer-name
    device_client : DeviceClient    # pylint: disable=redefined-outer-name
) -> None:
    # Load routing corp batch 5 rules for insertion
    descriptor_loader = DescriptorLoader(
        descriptors_file=DESC_FILE_RULES_INSERT_ROUTING_CORP, context_client=context_client, device_client=device_client
    )
    results = descriptor_loader.process()
    check_descriptor_load_results(results, descriptor_loader)

def test_rules_after_insertion_routing_corp_batch_5(
    context_client : ContextClient # pylint: disable=redefined-outer-name
) -> None:
    # State **after** inserting batch 5 of routing corp rules
    # Still 3 devices
    response = context_client.ListDevices(ADMIN_CONTEXT_ID)
    LOGGER.warning('Devices[{:d}] = {:s}'.format(len(response.devices), grpc_message_to_json_string(response)))
    assert len(response.devices) == DEV_NB

    # Verify that the following rules are in place
    desired_rules_nb = \
        CONNECTION_RULES + \
        ENDPOINT_RULES + \
        DATAPLANE_RULES_NB_INT_B1 + \
        DATAPLANE_RULES_NB_INT_B2 + \
        DATAPLANE_RULES_NB_INT_B3 + \
        DATAPLANE_RULES_NB_RT_EDGE + \
        DATAPLANE_RULES_NB_RT_CORP
    verify_number_of_rules(response.devices, desired_rules_nb)

def test_rules_insertion_acl_batch_6(
    context_client : ContextClient, # pylint: disable=redefined-outer-name
    device_client : DeviceClient    # pylint: disable=redefined-outer-name
) -> None:
    # Load ACL batch 6 rules for insertion
    descriptor_loader = DescriptorLoader(
        descriptors_file=DESC_FILE_RULES_INSERT_ACL, context_client=context_client, device_client=device_client
    )
    results = descriptor_loader.process()
    check_descriptor_load_results(results, descriptor_loader)

def test_rules_after_insertion_acl_batch_6(
    context_client : ContextClient # pylint: disable=redefined-outer-name
) -> None:
    # State **after** inserting batch 6 of ACL rules
    # Still 3 devices
    response = context_client.ListDevices(ADMIN_CONTEXT_ID)
    LOGGER.warning('Devices[{:d}] = {:s}'.format(len(response.devices), grpc_message_to_json_string(response)))
    assert len(response.devices) == DEV_NB

    # Verify that the following rules are in place
    desired_rules_nb = \
        CONNECTION_RULES + \
        ENDPOINT_RULES + \
        DATAPLANE_RULES_NB_INT_B1 + \
        DATAPLANE_RULES_NB_INT_B2 + \
        DATAPLANE_RULES_NB_INT_B3 + \
        DATAPLANE_RULES_NB_RT_EDGE + \
        DATAPLANE_RULES_NB_RT_CORP + \
        DATAPLANE_RULES_NB_ACL
    assert desired_rules_nb == CONNECTION_RULES + ENDPOINT_RULES + DATAPLANE_RULES_NB_TOT
    verify_number_of_rules(response.devices, desired_rules_nb)
