# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import operator
from sqlalchemy import Column, Enum, Float, ForeignKey, String
from typing import Dict
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy.orm import relationship
from .enums.ServiceStatus import ORM_ServiceStatusEnum
from .enums.ServiceType import ORM_ServiceTypeEnum
from ._Base import _Base

class ServiceModel(_Base):
    __tablename__ = 'service'

    context_uuid   = Column(UUID(as_uuid=False), ForeignKey('context.context_uuid'), primary_key=True)
    service_uuid   = Column(UUID(as_uuid=False), primary_key=True)
    service_name   = Column(String, nullable=False)
    service_type   = Column(Enum(ORM_ServiceTypeEnum))
    service_status = Column(Enum(ORM_ServiceStatusEnum))
    created_at     = Column(Float)

    context           = relationship('ContextModel', back_populates='services')
    service_endpoints = relationship('ServiceEndPointModel', back_populates='service') #, lazy='joined')
    #constraints       = relationship('ConstraintModel', passive_deletes=True, back_populates='service', lazy='joined')
    config_rules      = relationship('ConfigRuleModel', passive_deletes=True, back_populates='service', lazy='joined')

    def dump_id(self) -> Dict:
        return {
            'context_id': self.context.dump_id(),
            'service_uuid': {'uuid': self.service_uuid},
        }

    def dump(self) -> Dict:
        return {
            'service_id'          : self.dump_id(),
            'name'                : self.service_name,
            'service_type'        : self.service_type.value,
            'service_status'      : {'service_status': self.service_status.value},
            'service_endpoint_ids': [
                service_endpoint.endpoint.dump_id()
                for service_endpoint in self.service_endpoints
            ],
            'service_constraints' : [
                #constraint.dump()
                #for constraint in sorted(self.constraints, key=operator.attrgetter('position'))
            ],
            'service_config'      : {'config_rules': [
                config_rule.dump()
                for config_rule in sorted(self.config_rules, key=operator.attrgetter('position'))
            ]},
        }
