# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import enum, json
from sqlalchemy import Column, INTEGER, CheckConstraint, Enum, ForeignKeyConstraint, String, UniqueConstraint, text
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy.orm import relationship
from typing import Dict
from .enums.ConfigAction import ORM_ConfigActionEnum
from ._Base import _Base

# Enum values should match name of field in ConfigRuleModel
class ConfigRuleKindEnum(enum.Enum):
    CUSTOM = 'custom'
    ACL    = 'acl'

class ConfigRuleModel(_Base):
    __tablename__ = 'config_rule'

    config_rule_uuid = Column(UUID(as_uuid=False), primary_key=True, server_default=text('uuid_generate_v4()'))
    device_uuid      = Column(UUID(as_uuid=False)) # for device config rules
    context_uuid     = Column(UUID(as_uuid=False)) # for service/slice config rules
    service_uuid     = Column(UUID(as_uuid=False)) # for service config rules
    #slice_uuid       = Column(UUID(as_uuid=False)) # for slice config rules
    kind             = Column(Enum(ConfigRuleKindEnum))
    action           = Column(Enum(ORM_ConfigActionEnum))
    position         = Column(INTEGER, nullable=False)
    data             = Column(String, nullable=False)

    __table_args__ = (
        CheckConstraint(position >= 0, name='check_position_value'),
        UniqueConstraint('device_uuid', 'position', name='unique_per_device'),
        UniqueConstraint('context_uuid', 'service_uuid', 'position', name='unique_per_service'),
        #UniqueConstraint('context_uuid', 'slice_uuid', 'position', name='unique_per_slice'),
        ForeignKeyConstraint(
            ['device_uuid'],
            ['device.device_uuid'],
            ondelete='CASCADE'),
        ForeignKeyConstraint(
            ['context_uuid', 'service_uuid'],
            ['service.context_uuid', 'service.service_uuid'],
            ondelete='CASCADE'),
        #ForeignKeyConstraint(
        #    ['context_uuid', 'slice_uuid'],
        #    ['slice.context_uuid', 'slice.slice_uuid'],
        #    ondelete='CASCADE'),
    )

    device = relationship('DeviceModel', back_populates='config_rules')
    service = relationship('ServiceModel', back_populates='config_rules')
    #slice = relationship('SliceModel', back_populates='config_rules')

    def dump(self) -> Dict:
        return {self.kind.value: json.loads(self.data)}
