# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import anytree, json, logging
from typing import Any, Dict, List, Optional, Tuple, Union
from common.orm.Database import Database
from common.orm.HighLevel import get_object
from common.orm.backend.Tools import key_to_str
from common.proto.context_pb2 import Device
from common.tools.object_factory.ConfigRule import json_config_rule_delete, json_config_rule_set
from common.type_checkers.Checkers import chk_length, chk_type
from context.client.ContextClient import ContextClient
from device.client.DeviceClient import DeviceClient
from service.service.database.ConfigModel import ORM_ConfigActionEnum, get_config_rules
from service.service.database.ContextModel import ContextModel
from service.service.database.DeviceModel import DeviceModel
from service.service.database.ServiceModel import ServiceModel
from service.service.service_handler_api._ServiceHandler import _ServiceHandler
from service.service.service_handler_api.AnyTreeTools import TreeNode, delete_subnode, get_subnode, set_subnode_value

LOGGER = logging.getLogger(__name__)

class L3NMEmulatedServiceHandler(_ServiceHandler):
    def __init__(   # pylint: disable=super-init-not-called
        self, db_service : ServiceModel, database : Database, context_client : ContextClient,
        device_client : DeviceClient, **settings
    ) -> None:
        self.__db_service = db_service
        self.__database = database
        self.__context_client = context_client # pylint: disable=unused-private-member
        self.__device_client = device_client

        self.__db_context : ContextModel = get_object(self.__database, ContextModel, self.__db_service.context_fk)
        str_service_key = key_to_str([self.__db_context.context_uuid, self.__db_service.service_uuid])
        db_config = get_config_rules(self.__database, str_service_key, 'running')
        self.__resolver = anytree.Resolver(pathattr='name')
        self.__config = TreeNode('.')
        for action, resource_key, resource_value in db_config:
            if action == ORM_ConfigActionEnum.SET:
                try:
                    resource_value = json.loads(resource_value)
                except: # pylint: disable=bare-except
                    pass
                set_subnode_value(self.__resolver, self.__config, resource_key, resource_value)
            elif action == ORM_ConfigActionEnum.DELETE:
                delete_subnode(self.__resolver, self.__config, resource_key)

    def SetEndpoint(self, endpoints : List[Tuple[str, str, Optional[str]]]) -> List[Union[bool, Exception]]:
        chk_type('endpoints', endpoints, list)
        if len(endpoints) == 0: return []

        service_uuid              = self.__db_service.service_uuid
        service_short_uuid        = service_uuid.split('-')[-1]
        network_instance_name     = '{:s}-NetInst'.format(service_short_uuid)
        network_interface_desc    = '{:s}-NetIf'.format(service_uuid)
        network_subinterface_desc = '{:s}-NetSubIf'.format(service_uuid)

        settings : TreeNode = get_subnode(self.__resolver, self.__config, '/settings', None)
        if settings is None: raise Exception('Unable to retrieve service settings')
        json_settings : Dict = settings.value
        mtu                 = json_settings.get('mtu',                 1450 )    # 1512
        #address_families    = json_settings.get('address_families',    []   )    # ['IPV4']
        bgp_as              = json_settings.get('bgp_as',              0    )    # 65000
        bgp_route_target    = json_settings.get('bgp_route_target',    '0:0')    # 65000:333

        results = []
        for endpoint in endpoints:
            try:
                chk_type('endpoint', endpoint, (tuple, list))
                chk_length('endpoint', endpoint, min_length=2, max_length=3)
                if len(endpoint) == 2:
                    device_uuid, endpoint_uuid = endpoint
                else:
                    device_uuid, endpoint_uuid, _ = endpoint # ignore topology_uuid by now

                endpoint_settings_uri = '/device[{:s}]/endpoint[{:s}]/settings'.format(device_uuid, endpoint_uuid)
                endpoint_settings : TreeNode = get_subnode(self.__resolver, self.__config, endpoint_settings_uri, None)
                if endpoint_settings is None:
                    raise Exception('Unable to retrieve service settings for endpoint({:s})'.format(
                        str(endpoint_settings_uri)))
                json_endpoint_settings : Dict = endpoint_settings.value
                #router_id           = json_endpoint_settings.get('router_id',           '0.0.0.0')  # '10.95.0.10'
                route_distinguisher = json_endpoint_settings.get('route_distinguisher', '0:0'    )  # '60001:801'
                sub_interface_index = json_endpoint_settings.get('sub_interface_index', 0        )  # 1
                vlan_id             = json_endpoint_settings.get('vlan_id',             1        )  # 400
                address_ip          = json_endpoint_settings.get('address_ip',          '0.0.0.0')  # '2.2.2.1'
                address_prefix      = json_endpoint_settings.get('address_prefix',      24       )  # 30
                if_subif_name       = '{:s}.{:d}'.format(endpoint_uuid, vlan_id)

                db_device : DeviceModel = get_object(self.__database, DeviceModel, device_uuid, raise_if_not_found=True)
                json_device = db_device.dump(include_config_rules=False, include_drivers=True, include_endpoints=True)
                json_device_config : Dict = json_device.setdefault('device_config', {})
                json_device_config_rules : List = json_device_config.setdefault('config_rules', [])
                json_device_config_rules.extend([
                    json_config_rule_set(
                        '/network_instance[{:s}]'.format(network_instance_name), {
                            'name': network_instance_name, 'description': network_interface_desc, 'type': 'L3VRF',
                            'route_distinguisher': route_distinguisher,
                            #'router_id': router_id, 'address_families': address_families,
                    }),
                    json_config_rule_set(
                        '/interface[{:s}]'.format(endpoint_uuid), {
                            'name': endpoint_uuid, 'description': network_interface_desc, 'mtu': mtu,
                    }),
                    json_config_rule_set(
                        '/interface[{:s}]/subinterface[{:d}]'.format(endpoint_uuid, sub_interface_index), {
                            'name': endpoint_uuid, 'index': sub_interface_index,
                            'description': network_subinterface_desc, 'vlan_id': vlan_id,
                            'address_ip': address_ip, 'address_prefix': address_prefix,
                    }),
                    json_config_rule_set(
                        '/network_instance[{:s}]/interface[{:s}]'.format(network_instance_name, if_subif_name), {
                            'name': network_instance_name, 'id': if_subif_name, 'interface': endpoint_uuid,
                            'subinterface': sub_interface_index,
                    }),
                    json_config_rule_set(
                        '/network_instance[{:s}]/protocols[BGP]'.format(network_instance_name), {
                            'name': network_instance_name, 'identifier': 'BGP', 'protocol_name': 'BGP', 'as': bgp_as,
                    }),
                    json_config_rule_set(
                        '/network_instance[{:s}]/table_connections[STATIC][BGP][IPV4]'.format(network_instance_name), {
                            'name': network_instance_name, 'src_protocol': 'STATIC', 'dst_protocol': 'BGP',
                            'address_family': 'IPV4', #'default_import_policy': 'REJECT_ROUTE',
                    }),
                    json_config_rule_set(
                        '/network_instance[{:s}]/table_connections[DIRECTLY_CONNECTED][BGP][IPV4]'.format(
                            network_instance_name), {
                            'name': network_instance_name, 'src_protocol': 'DIRECTLY_CONNECTED', 'dst_protocol': 'BGP',
                            'address_family': 'IPV4', #'default_import_policy': 'REJECT_ROUTE',
                    }),
                    json_config_rule_set(
                        '/routing_policy/bgp_defined_set[{:s}_rt_import]'.format(network_instance_name), {
                            'ext_community_set_name': '{:s}_rt_import'.format(network_instance_name),
                    }),
                    json_config_rule_set(
                        '/routing_policy/bgp_defined_set[{:s}_rt_import][route-target:{:s}]'.format(
                            network_instance_name, bgp_route_target), {
                            'ext_community_set_name': '{:s}_rt_import'.format(network_instance_name),
                            'ext_community_member'  : 'route-target:{:s}'.format(bgp_route_target),
                    }),
                    json_config_rule_set(
                        '/routing_policy/policy_definition[{:s}_import]'.format(network_instance_name), {
                            'policy_name': '{:s}_import'.format(network_instance_name),
                    }),
                    json_config_rule_set(
                        '/routing_policy/policy_definition[{:s}_import]/statement[{:s}]'.format(
                            network_instance_name, '3'), {
                            'policy_name': '{:s}_import'.format(network_instance_name), 'statement_name': '3',
                            'ext_community_set_name': '{:s}_rt_import'.format(network_instance_name),
                            'match_set_options': 'ANY', 'policy_result': 'ACCEPT_ROUTE',
                    }),
                    json_config_rule_set(
                        # pylint: disable=duplicate-string-formatting-argument
                        '/network_instance[{:s}]/inter_instance_policies[{:s}_import]'.format(
                            network_instance_name, network_instance_name), {
                            'name': network_instance_name, 'import_policy': '{:s}_import'.format(network_instance_name),
                    }),
                    json_config_rule_set(
                        '/routing_policy/bgp_defined_set[{:s}_rt_export]'.format(network_instance_name), {
                            'ext_community_set_name': '{:s}_rt_export'.format(network_instance_name),
                    }),
                    json_config_rule_set(
                        '/routing_policy/bgp_defined_set[{:s}_rt_export][route-target:{:s}]'.format(
                            network_instance_name, bgp_route_target), {
                            'ext_community_set_name': '{:s}_rt_export'.format(network_instance_name),
                            'ext_community_member'  : 'route-target:{:s}'.format(bgp_route_target),
                    }),
                    json_config_rule_set(
                        '/routing_policy/policy_definition[{:s}_export]'.format(network_instance_name), {
                            'policy_name': '{:s}_export'.format(network_instance_name),
                    }),
                    json_config_rule_set(
                        '/routing_policy/policy_definition[{:s}_export]/statement[{:s}]'.format(
                            network_instance_name, '3'), {
                            'policy_name': '{:s}_export'.format(network_instance_name), 'statement_name': '3',
                            'ext_community_set_name': '{:s}_rt_export'.format(network_instance_name),
                            'match_set_options': 'ANY', 'policy_result': 'ACCEPT_ROUTE',
                    }),
                    json_config_rule_set(
                        # pylint: disable=duplicate-string-formatting-argument
                        '/network_instance[{:s}]/inter_instance_policies[{:s}_export]'.format(
                            network_instance_name, network_instance_name), {
                            'name': network_instance_name, 'export_policy': '{:s}_export'.format(network_instance_name),
                    }),
                ])
                self.__device_client.ConfigureDevice(Device(**json_device))
                results.append(True)
            except Exception as e: # pylint: disable=broad-except
                LOGGER.exception('Unable to SetEndpoint({:s})'.format(str(endpoint)))
                results.append(e)

        return results

    def DeleteEndpoint(self, endpoints : List[Tuple[str, str, Optional[str]]]) -> List[Union[bool, Exception]]:
        chk_type('endpoints', endpoints, list)
        if len(endpoints) == 0: return []

        service_uuid              = self.__db_service.service_uuid
        service_short_uuid        = service_uuid.split('-')[-1]
        network_instance_name     = '{:s}-NetInst'.format(service_short_uuid)

        settings : TreeNode = get_subnode(self.__resolver, self.__config, '/settings', None)
        if settings is None: raise Exception('Unable to retrieve service settings')
        json_settings : Dict = settings.value
        bgp_route_target    = json_settings.get('bgp_route_target',    '0:0')    # 65000:333

        results = []
        for endpoint in endpoints:
            try:
                chk_type('endpoint', endpoint, (tuple, list))
                chk_length('endpoint', endpoint, min_length=2, max_length=3)
                if len(endpoint) == 2:
                    device_uuid, endpoint_uuid = endpoint
                else:
                    device_uuid, endpoint_uuid, _ = endpoint # ignore topology_uuid by now

                endpoint_settings_uri = '/device[{:s}]/endpoint[{:s}]/settings'.format(device_uuid, endpoint_uuid)
                endpoint_settings : TreeNode = get_subnode(self.__resolver, self.__config, endpoint_settings_uri, None)
                if endpoint_settings is None:
                    raise Exception('Unable to retrieve service settings for endpoint({:s})'.format(
                        str(endpoint_settings_uri)))
                json_endpoint_settings : Dict = endpoint_settings.value
                sub_interface_index = json_endpoint_settings.get('sub_interface_index', 0        )  # 1
                vlan_id             = json_endpoint_settings.get('vlan_id',             1        )  # 400
                if_subif_name       = '{:s}.{:d}'.format(endpoint_uuid, vlan_id)

                db_device : DeviceModel = get_object(self.__database, DeviceModel, device_uuid, raise_if_not_found=True)
                json_device = db_device.dump(include_config_rules=False, include_drivers=True, include_endpoints=True)
                json_device_config : Dict = json_device.setdefault('device_config', {})
                json_device_config_rules : List = json_device_config.setdefault('config_rules', [])
                json_device_config_rules.extend([
                    json_config_rule_delete(
                        '/network_instance[{:s}]/interface[{:s}]'.format(network_instance_name, if_subif_name), {
                            'name': network_instance_name, 'id': if_subif_name,
                    }),
                    json_config_rule_delete(
                        '/interface[{:s}]/subinterface[{:d}]'.format(endpoint_uuid, sub_interface_index), {
                            'name': endpoint_uuid, 'index': sub_interface_index,
                    }),
                    json_config_rule_delete(
                        '/interface[{:s}]'.format(endpoint_uuid), {
                            'name': endpoint_uuid,
                    }),
                    json_config_rule_delete(
                        '/network_instance[{:s}]/table_connections[DIRECTLY_CONNECTED][BGP][IPV4]'.format(
                            network_instance_name), {
                            'name': network_instance_name, 'src_protocol': 'DIRECTLY_CONNECTED', 'dst_protocol': 'BGP',
                            'address_family': 'IPV4',
                    }),
                    json_config_rule_delete(
                        '/network_instance[{:s}]/table_connections[STATIC][BGP][IPV4]'.format(network_instance_name), {
                            'name': network_instance_name, 'src_protocol': 'STATIC', 'dst_protocol': 'BGP',
                            'address_family': 'IPV4',
                    }),
                    json_config_rule_delete(
                        '/network_instance[{:s}]/protocols[BGP]'.format(network_instance_name), {
                            'name': network_instance_name, 'identifier': 'BGP', 'protocol_name': 'BGP',
                    }),
                    json_config_rule_delete(
                        # pylint: disable=duplicate-string-formatting-argument
                        '/network_instance[{:s}]/inter_instance_policies[{:s}_import]'.format(
                            network_instance_name, network_instance_name), {
                        'name': network_instance_name,
                    }),
                    json_config_rule_delete(
                        '/routing_policy/policy_definition[{:s}_import]/statement[{:s}]'.format(
                            network_instance_name, '3'), {
                            'policy_name': '{:s}_import'.format(network_instance_name), 'statement_name': '3',
                    }),
                    json_config_rule_delete(
                        '/routing_policy/policy_definition[{:s}_import]'.format(network_instance_name), {
                            'policy_name': '{:s}_import'.format(network_instance_name),
                    }),
                    json_config_rule_delete(
                        '/routing_policy/bgp_defined_set[{:s}_rt_import][route-target:{:s}]'.format(
                            network_instance_name, bgp_route_target), {
                            'ext_community_set_name': '{:s}_rt_import'.format(network_instance_name),
                            'ext_community_member'  : 'route-target:{:s}'.format(bgp_route_target),
                    }),
                    json_config_rule_delete(
                        '/routing_policy/bgp_defined_set[{:s}_rt_import]'.format(network_instance_name), {
                            'ext_community_set_name': '{:s}_rt_import'.format(network_instance_name),
                    }),
                    json_config_rule_delete(
                        # pylint: disable=duplicate-string-formatting-argument
                        '/network_instance[{:s}]/inter_instance_policies[{:s}_export]'.format(
                            network_instance_name, network_instance_name), {
                            'name': network_instance_name,
                    }),
                    json_config_rule_delete(
                        '/routing_policy/policy_definition[{:s}_export]/statement[{:s}]'.format(
                            network_instance_name, '3'), {
                            'policy_name': '{:s}_export'.format(network_instance_name), 'statement_name': '3',
                    }),
                    json_config_rule_delete(
                        '/routing_policy/policy_definition[{:s}_export]'.format(network_instance_name), {
                            'policy_name': '{:s}_export'.format(network_instance_name),
                    }),
                    json_config_rule_delete(
                        '/routing_policy/bgp_defined_set[{:s}_rt_export][route-target:{:s}]'.format(
                            network_instance_name, bgp_route_target), {
                            'ext_community_set_name': '{:s}_rt_export'.format(network_instance_name),
                            'ext_community_member'  : 'route-target:{:s}'.format(bgp_route_target),
                    }),
                    json_config_rule_delete(
                        '/routing_policy/bgp_defined_set[{:s}_rt_export]'.format(network_instance_name), {
                            'ext_community_set_name': '{:s}_rt_export'.format(network_instance_name),
                    }),
                    json_config_rule_delete(
                        '/network_instance[{:s}]'.format(network_instance_name), {
                            'name': network_instance_name
                    }),
                ])
                self.__device_client.ConfigureDevice(Device(**json_device))
                results.append(True)
            except Exception as e: # pylint: disable=broad-except
                LOGGER.exception('Unable to DeleteEndpoint({:s})'.format(str(endpoint)))
                results.append(e)

        return results

    def SetConstraint(self, constraints : List[Tuple[str, Any]]) -> List[Union[bool, Exception]]:
        chk_type('constraints', constraints, list)
        if len(constraints) == 0: return []

        msg = '[SetConstraint] Method not implemented. Constraints({:s}) are being ignored.'
        LOGGER.warning(msg.format(str(constraints)))
        return [True for _ in range(len(constraints))]

    def DeleteConstraint(self, constraints : List[Tuple[str, Any]]) -> List[Union[bool, Exception]]:
        chk_type('constraints', constraints, list)
        if len(constraints) == 0: return []

        msg = '[DeleteConstraint] Method not implemented. Constraints({:s}) are being ignored.'
        LOGGER.warning(msg.format(str(constraints)))
        return [True for _ in range(len(constraints))]

    def SetConfig(self, resources : List[Tuple[str, Any]]) -> List[Union[bool, Exception]]:
        chk_type('resources', resources, list)
        if len(resources) == 0: return []

        results = []
        for resource in resources:
            try:
                resource_key, resource_value = resource
                resource_value = json.loads(resource_value)
                set_subnode_value(self.__resolver, self.__config, resource_key, resource_value)
                results.append(True)
            except Exception as e: # pylint: disable=broad-except
                LOGGER.exception('Unable to SetConfig({:s})'.format(str(resource)))
                results.append(e)

        return results

    def DeleteConfig(self, resources : List[Tuple[str, Any]]) -> List[Union[bool, Exception]]:
        chk_type('resources', resources, list)
        if len(resources) == 0: return []

        results = []
        for resource in resources:
            try:
                resource_key, _ = resource
                delete_subnode(self.__resolver, self.__config, resource_key)
            except Exception as e: # pylint: disable=broad-except
                LOGGER.exception('Unable to DeleteConfig({:s})'.format(str(resource)))
                results.append(e)

        return results
