# Copyright 2022-2024 ETSI OSG/SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import logging
import sqlalchemy_utils
from sqlalchemy.orm import sessionmaker
from kpi_manager.database.KpiEngine import KpiEngine
from kpi_manager.database.KpiModel import Kpi as KpiModel
from common.method_wrappers.ServiceExceptions import ( 
    AlreadyExistsException, OperationFailedException , NotFoundException)

LOGGER = logging.getLogger(__name__)
DB_NAME = "tfs_kpi_mgmt"

class KpiDB:
    def __init__(self):
        self.db_engine = KpiEngine.get_engine()
        if self.db_engine is None:
            LOGGER.error('Unable to get SQLAlchemy DB Engine...')
            return False
        self.db_name = DB_NAME
        self.Session = sessionmaker(bind=self.db_engine)

    def create_database(self) -> None:
        if not sqlalchemy_utils.database_exists(self.db_engine.url):
            sqlalchemy_utils.create_database(self.db_engine.url)
            LOGGER.debug("Database created. {:}".format(self.db_engine.url))

    def drop_database(self) -> None:
        if sqlalchemy_utils.database_exists(self.db_engine.url):
            sqlalchemy_utils.drop_database(self.db_engine.url)

    def create_tables(self):
        # TODO: use "get_tables(declatrative class obj)" method of "sqlalchemy_utils" to verify tables.
        try:
            KpiModel.metadata.create_all(self.db_engine)     # type: ignore
            LOGGER.debug("Tables created in the DB Name: {:}".format(self.db_name))
        except Exception as e:
            LOGGER.debug("Tables cannot be created in the kpi database. {:s}".format(str(e)))
            raise OperationFailedException ("Tables can't be created", extra_details=["unable to create table {:}".format(e)])

    def verify_tables(self):
        try:
            with self.db_engine.connect() as connection:
                result = connection.execute("SHOW TABLES;")
                tables = result.fetchall()      # type: ignore
                LOGGER.debug("Tables verified: {:}".format(tables))
        except Exception as e:
            LOGGER.debug("Unable to fetch Table names. {:s}".format(str(e)))

    def add_row_to_db(self, row):
        session = self.Session()
        try:
            session.add(row)
            session.commit()
            LOGGER.debug(f"Row inserted into {row.__class__.__name__} table.")
            return True
        except Exception as e:
            session.rollback()
            if "psycopg2.errors.UniqueViolation" in str(e):
                LOGGER.error(f"Unique key voilation: {row.__class__.__name__} table. {str(e)}")
                raise AlreadyExistsException(row.__class__.__name__, row, extra_details=["Unique key voilation: {:}".format(e)] )
            else:
                LOGGER.error(f"Failed to insert new row into {row.__class__.__name__} table. {str(e)}")
                raise OperationFailedException ("Deletion by column id", extra_details=["unable to delete row {:}".format(e)])
        finally:
            session.close()
    
    def search_db_row_by_id(self, model, col_name, id_to_search):
        session = self.Session()
        try:
            entity = session.query(model).filter_by(**{col_name: id_to_search}).first()
            if entity:
                # LOGGER.debug(f"{model.__name__} ID found: {str(entity)}")
                return entity
            else:
                LOGGER.debug(f"{model.__name__} ID not found, No matching row: {str(id_to_search)}")
                print("{:} ID not found, No matching row: {:}".format(model.__name__, id_to_search))
                return None
        except Exception as e:
            LOGGER.debug(f"Failed to retrieve {model.__name__} ID. {str(e)}")
            raise OperationFailedException ("search by column id", extra_details=["unable to search row {:}".format(e)])
        finally:
            session.close()
    
    def delete_db_row_by_id(self, model, col_name, id_to_search):
        session = self.Session()
        try:
            record = session.query(model).filter_by(**{col_name: id_to_search}).first()
            if record:
                session.delete(record)
                session.commit()
                LOGGER.debug("Deleted %s with %s: %s", model.__name__, col_name, id_to_search)
            else:
                LOGGER.debug("%s with %s %s not found", model.__name__, col_name, id_to_search)
                return None
        except Exception as e:
            session.rollback()
            LOGGER.error("Error deleting %s with %s %s: %s", model.__name__, col_name, id_to_search, e)
            raise OperationFailedException ("Deletion by column id", extra_details=["unable to delete row {:}".format(e)])
        finally:
            session.close()

    def select_with_filter(self, model, filter_object):
        session = self.Session()
        try:
            query = session.query(KpiModel)
            # Apply filters based on the filter_object
            if filter_object.kpi_id:
                query = query.filter(KpiModel.kpi_id.in_([k.kpi_id.uuid for k in filter_object.kpi_id]))

            if filter_object.kpi_sample_type:
                query = query.filter(KpiModel.kpi_sample_type.in_(filter_object.kpi_sample_type))

            if filter_object.device_id:
                query = query.filter(KpiModel.device_id.in_([d.device_uuid.uuid for d in filter_object.device_id]))

            if filter_object.endpoint_id:
                query = query.filter(KpiModel.endpoint_id.in_([e.endpoint_uuid.uuid for e in filter_object.endpoint_id]))

            if filter_object.service_id:
                query = query.filter(KpiModel.service_id.in_([s.service_uuid.uuid for s in filter_object.service_id]))

            if filter_object.slice_id:
                query = query.filter(KpiModel.slice_id.in_([s.slice_uuid.uuid for s in filter_object.slice_id]))

            if filter_object.connection_id:
                query = query.filter(KpiModel.connection_id.in_([c.connection_uuid.uuid for c in filter_object.connection_id]))

            if filter_object.link_id:
                query = query.filter(KpiModel.link_id.in_([l.link_uuid.uuid for l in filter_object.link_id]))
            result = query.all()
            
            if result:
                LOGGER.debug(f"Fetched filtered rows from {model.__name__} table with filters: {filter_object}") #  - Results: {result}
            else:
                LOGGER.debug(f"No matching row found in {model.__name__} table with filters: {filter_object}")
            return result
        except Exception as e:
            LOGGER.error(f"Error fetching filtered rows from {model.__name__} table with filters {filter_object} ::: {e}")
            raise OperationFailedException ("Select by filter", extra_details=["unable to apply the filter {:}".format(e)])
        finally:
            session.close()
