import asyncio, logging
from typing import Dict

LOGGER = logging.getLogger(__name__)

class EventSequence:
    # all values expressed as float in seconds

    def __init__(self, name, event_rate, start_delay=0, duration=0, num_repeats=0) -> None:
        self.name        = name        # name of this event sequence
        self.start_delay = start_delay # seconds to wait before first event; 0 means trigger first event immediately
        self.duration    = duration    # seconds the event sequence should elapse, includes wait times, 0 means infinity
        self.num_repeats = num_repeats # number of times the event should be triggered, 0 means infinity
        self.event_rate  = event_rate  # wait time between events, must be positive float

        # Internal variables:
        self.__next_delay       = self.start_delay  # delay until next event
        self.__termination_time = None              # termination time, if duration > 0, else None and runs infinity
        self.__terminate        = asyncio.Event()   # do not execute more iterations when terminate is set

    def schedule(self):
        loop = asyncio.get_event_loop()
        current_time = loop.time()

        if (self.num_repeats == 0) and (self.duration > 0):
            self.__termination_time = current_time + self.duration

        if self.__termination_time and (current_time > self.__termination_time): return

        LOGGER.info('Scheduling {} for #{} time to run after {} seconds...'.format(
            self.name, self.num_repeats, self.__next_delay))
        loop.call_later(self.__next_delay, self.run)
        self.__next_delay = self.event_rate

    def terminate(self): self.__terminate.set()
    def terminate_is_set(self): return self.__terminate.is_set

    def run(self):
        if self.terminate_is_set(): return
        LOGGER.info('Running {} for #{} time...'.format(self.name, self.num_repeats))
        self.schedule()
        self.num_repeats += 1

class DiscreteEventSimulator:
    def __init__(self) -> None:
        self.__eventsequences : Dict[str, EventSequence] = {}

    def add_event_sequence(self, event_sequence : EventSequence):
        if event_sequence.name in self.__eventsequences: return
        self.__eventsequences[event_sequence.name] = event_sequence
        event_sequence.schedule()
        return event_sequence

    def remove_event_sequence(self, event_sequence : EventSequence):
        if event_sequence.name not in self.__eventsequences: return
        event_sequence = self.__eventsequences.pop(event_sequence.name)
        event_sequence.terminate()

async def terminate(des : DiscreteEventSimulator, es: EventSequence):
    des.remove_event_sequence(es)

async def async_main():
    des = DiscreteEventSimulator()
    es1 = des.add_event_sequence(EventSequence('ES1-10s', 1.0, start_delay=3.0, duration=10.0))
    es2 = des.add_event_sequence(EventSequence('ES2-inf', 1.5, start_delay=3.0))
    es3 = des.add_event_sequence(EventSequence('ES3-10r', 2.0, num_repeats=5))
    loop = asyncio.get_event_loop()
    loop.call_later(60.0, terminate, des, es3)

if __name__ == '__main__':
    loop = asyncio.get_event_loop()
    loop.run_until_complete(async_main())
