from typing import Any, Iterator, List, Tuple, Union

class _Driver:
    def __init__(self, address : str, port : int, **kwargs) -> None:
        """ Initialize Driver.
            Parameters:
                address : str
                    The address of the device
                port : int
                    The port of the device
                **kwargs
                    Extra attributes can be configured using kwargs.
        """
        raise NotImplementedError()

    def Connect(self) -> bool:
        """ Connect to the Device.
            Returns:
                succeeded : bool
                    Boolean variable indicating if connection succeeded.
        """
        raise NotImplementedError()

    def Disconnect(self) -> bool:
        """ Disconnect from the Device.
            Returns:
                succeeded : bool
                    Boolean variable indicating if disconnection succeeded.
        """
        raise NotImplementedError()

    def GetConfig(self, resources : List[str]) -> List[Union[Any, None]]:
        """ Retrieve running configuration of entire device, or selected resources.
            Parameters:
                resources : List[str]
                    List of keys pointing to the resources to be retrieved.
            Returns:
                values : List[Union[Any, None]]
                    List of values for resource keys requested. Values should be in the same order than resource keys
                    requested. If a resource is not found, None should be specified in the List for that resource.
        """
        raise NotImplementedError()

    def SetConfig(self, resources : List[Tuple[str, Any]]) -> List[bool]:
        """ Create/Update configuration for a list of resources.
            Parameters:
                resources : List[Tuple[str, Any]]
                    List of tuples, each containing a resource_key pointing the resource to be modified, and a
                    resource_value containing the new value to be set.
            Returns:
                results : List[bool]
                    List of results for changes in resource keys requested. Each result is a boolean value indicating
                    if the change succeeded. Results should be in the same order than resource keys requested.
        """
        raise NotImplementedError()

    def DeleteConfig(self, resources : List[str]) -> List[bool]:
        """ Delete configuration for a list of resources.
            Parameters:
                resources : List[str]
                    List of keys pointing to the resources to be deleted.
            Returns:
                results : List[bool]
                    List of results for delete resource keys requested. Each result is a boolean value indicating
                    if the delete succeeded. Results should be in the same order than resource keys requested.
        """
        raise NotImplementedError()

    def SubscribeState(self, resources : List[Tuple[str, float]]) -> List[bool]:
        """ Subscribe to state information of entire device, or selected resources. Subscriptions are incremental.
            Driver should keep track of requested resources.
            Parameters:
                resources : List[Tuple[str, float]]
                    List of tuples, each containing a resource_key pointing the resource to be subscribed, and a
                    sampling_rate in seconds defining the desired monitoring periodicity for that resource.
                    Note: sampling_rate values must be a strictly positive floats.
            Returns:
                results : List[bool]
                    List of results for subscriptions to resource keys requested. Each result is a boolean value
                    indicating if the subscription succeeded. Results should be in the same order than resource keys
                    requested.
        """
        raise NotImplementedError()

    def UnsubscribeState(self, resources : List[str]) -> List[bool]:
        """ Unsubscribe from state information of entire device, or selected resources. Subscriptions are incremental.
            Driver should keep track of requested resources.
            Parameters:
                resources : List[str]
                    List of resource_keys pointing the resource to be unsubscribed.
            Returns:
                results : List[bool]
                    List of results for unsubscriptions from resource keys requested. Each result is a boolean value
                    indicating if the unsubscription succeeded. Results should be in the same order than resource keys
                    requested.
        """
        raise NotImplementedError()

    def GetState(self) -> Iterator[Tuple[str, Any]]:
        """ Retrieve last collected values for subscribed resources. Operates as a generator, so this method should be
            called once and will block until values are available. When values are available, it should yield each of
            them and block again until new values are available. When the driver is destroyed, GetState() can return
            instead of yield to terminate the loop.
            Example:
                for resource_key,resource_value in my_driver.GetState():
                    process(resource_key,resource_value)
            Returns:
                results : Iterator[Tuple[str, Any]]
                    List of tuples with state samples, each containing a resource_key and a resource_value. Only
                    resources with an active subscription must be retrieved. Periodicity of the samples is specified
                    when creating the subscription using method SubscribeState(). Order of yielded values is arbitrary.
        """
        raise NotImplementedError()
