import logging, signal, sys, threading
from prometheus_client import start_http_server
from common.Settings import get_setting
from common.database.Factory import get_database
from context.Config import GRPC_SERVICE_PORT, GRPC_MAX_WORKERS, GRPC_GRACE_PERIOD, LOG_LEVEL, RESTAPI_SERVICE_PORT, \
    RESTAPI_BASE_URL, METRICS_PORT
from context.service.ContextService import ContextService
from context.service.rest_server.Server import Server
from context.service.rest_server.resources.Context import Context

terminate = threading.Event()
logger = None

def signal_handler(signal, frame):
    global terminate, logger
    logger.warning('Terminate signal received')
    terminate.set()

def main():
    global terminate, logger

    grpc_service_port    = get_setting('CONTEXTSERVICE_SERVICE_PORT_GRPC', default=GRPC_SERVICE_PORT   )
    max_workers          = get_setting('MAX_WORKERS',                      default=GRPC_MAX_WORKERS    )
    grace_period         = get_setting('GRACE_PERIOD',                     default=GRPC_GRACE_PERIOD   )
    log_level            = get_setting('LOG_LEVEL',                        default=LOG_LEVEL           )
    restapi_service_port = get_setting('RESTAPI_SERVICE_PORT',             default=RESTAPI_SERVICE_PORT)
    restapi_base_url     = get_setting('RESTAPI_BASE_URL',                 default=RESTAPI_BASE_URL    )
    metrics_port         = get_setting('METRICS_PORT',                     default=METRICS_PORT        )

    logging.basicConfig(level=log_level)
    logger = logging.getLogger(__name__)

    signal.signal(signal.SIGINT,  signal_handler)
    signal.signal(signal.SIGTERM, signal_handler)

    logger.info('Starting...')

    # Start metrics server
    start_http_server(metrics_port)

    # Get database instance
    database = get_database()

    # Starting context service
    grpc_service = ContextService(database, port=grpc_service_port, max_workers=max_workers, grace_period=grace_period)
    grpc_service.start()

    rest_server = Server(port=restapi_service_port, base_url=restapi_base_url)
    rest_server.add_resource(
        Context, '/restconf/config/context', endpoint='api.context', resource_class_args=(database,))
    rest_server.start()

    # Wait for Ctrl+C or termination signal
    while not terminate.wait(timeout=0.1): pass

    logger.info('Terminating...')
    grpc_service.stop()
    rest_server.shutdown()
    rest_server.join()

    logger.info('Bye')
    return 0

if __name__ == '__main__':
    sys.exit(main())
