# Copyright 2022-2023 ETSI TeraFlowSDN - TFS OSG (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# 

import json
import requests
import uuid
from common.Constants import *
from typing import List
from common.proto.context_pb2 import(
    Device, DeviceId, Service, Connection, EndPointId, TopologyId, ContextId, Uuid,
    ConfigRule, ConfigActionEnum, ConfigRule_Custom
)
from common.proto.pathcomp_pb2 import PathCompReply
from common.Settings import (
    ENVVAR_SUFIX_SERVICE_HOST, ENVVAR_SUFIX_SERVICE_PORT_GRPC, find_environment_variables, get_env_var_name
)
from service.service.tools.replies import reply_uni_txt, optical_band_uni_txt, reply_bid_txt, optical_band_bid_txt

log = logging.getLogger(__name__)

testing = False

VAR_NAME_OPTICAL_CONTROLLER_HOST = get_env_var_name(ServiceNameEnum.OPTICALCONTROLLER, ENVVAR_SUFIX_SERVICE_HOST)
VAR_NAME_OPTICAL_CONTROLLER_PORT = get_env_var_name(ServiceNameEnum.OPTICALCONTROLLER, ENVVAR_SUFIX_SERVICE_PORT_GRPC)

opticalcontrollers_url = find_environment_variables([
    VAR_NAME_OPTICAL_CONTROLLER_HOST,
    VAR_NAME_OPTICAL_CONTROLLER_PORT,
])
OPTICAL_IP   = opticalcontrollers_url.get(VAR_NAME_OPTICAL_CONTROLLER_HOST)
OPTICAL_PORT = opticalcontrollers_url.get(VAR_NAME_OPTICAL_CONTROLLER_PORT)
log.info(f"OPTICAL_IP:{OPTICAL_IP} OPTICAL_PORT:{OPTICAL_PORT}")

def get_uuids_from_names(devices: List[Device], device_name: str, port_name: str):
    device_uuid = ""
    port_uuid = ""
    for device in devices:
        if device.name == device_name:
            device_uuid  = device.device_id.device_uuid.uuid
            for ep in device.device_endpoints:
                if ep.name == port_name:
                    port_uuid = ep.endpoint_id.endpoint_uuid.uuid
                    return device_uuid, port_uuid
    return "", ""


def get_names_from_uuids(devices: List[Device], device_uuid: str, port_uuid: str):
    device_name = ""
    port_name = ""
    for device in devices:
        if device.device_id.device_uuid.uuid == device_uuid:
            device_name  = device.name
            for ep in device.device_endpoints:
                if ep.endpoint_id.endpoint_uuid.uuid == port_uuid:
                    port_name = ep.name
                    return device_name, port_name
    return "", ""

def get_device_name_from_uuid(devices: List[Device], device_uuid: str):
    device_name = ""
    
    for device in devices:
        if device.device_id.device_uuid.uuid == device_uuid:
            device_name  = device.name
            return device_name
    return ""

def refresh_opticalcontroller (topology_id:TopologyId):
     headers = {"Content-Type": "application/json"}
     urlx = "http://{}:{}/GetTopology/{}".format(OPTICAL_IP, OPTICAL_PORT,topology_id)
     res = requests.get(urlx, headers=headers)
     logging.info(f"Refresh opticalcontroller {res}")

def add_lightpath(src, dst, bitrate, bidir, ob_band) -> str:
    if not testing:
        urlx = ""
        headers = {"Content-Type": "application/json"}
        if ob_band is None:
            if bidir is None:
                bidir = 1
            urlx = "http://{}:{}/OpticalTFS/AddFlexLightpath/{}/{}/{}/{}".format(OPTICAL_IP, OPTICAL_PORT, src, dst, bitrate, bidir)
        else:
            if bidir is None:
                bidir = 1
            urlx = "http://{}:{}/OpticalTFS/AddFlexLightpath/{}/{}/{}/{}/{}".format(OPTICAL_IP, OPTICAL_PORT, src, dst, bitrate, bidir, ob_band)
        r = requests.put(urlx, headers=headers)
        print(f"addpathlight {r}")
        reply = r.text 
        return reply
    else:
        if bidir is not None:
            if bidir == 0:        
                return reply_uni_txt
        return reply_bid_txt
                

def get_optical_band(idx) -> str:
    if not testing:
        urlx = "http://{}:{}/OpticalTFS/GetOpticalBand/{}".format(OPTICAL_IP, OPTICAL_PORT, idx)
        headers = {"Content-Type": "application/json"}
        r = requests.get(urlx, headers=headers)
        reply = r.text 
        return reply
    else:
        if str(idx) == "1":
            return optical_band_bid_txt
        else:
            return optical_band_uni_txt

    
def delete_lightpath(flow_id, src, dst, bitrate) -> str:
    reply = "200"
    if not testing:
        urlx = "http://{}:{}/OpticalTFS/DelLightpath/{}/{}/{}/{}".format(OPTICAL_IP, OPTICAL_PORT, flow_id, src, dst, bitrate)

        headers = {"Content-Type": "application/json"}
        r = requests.delete(urlx, headers=headers)
        reply = r.text 
    return reply


def get_lightpaths() -> str:
    urlx = "http://{}:{}/OpticalTFS/GetLightpaths".format(OPTICAL_IP, OPTICAL_PORT)

    headers = {"Content-Type": "application/json"}
    r = requests.get(urlx, headers=headers)
    reply = r.text 
    return reply

        
def adapt_reply(devices, service, reply_json, context_id, topology_id, optical_band_txt) ->  PathCompReply:
    opt_reply = PathCompReply()
    topo = TopologyId(context_id=ContextId(context_uuid=Uuid(uuid=context_id)),topology_uuid=Uuid(uuid=topology_id))
    #add optical band connection first
    rules_ob= []
    ob_id = 0
    connection_ob=None
    if optical_band_txt != "":
        ob_json = json.loads(optical_band_txt)
        ob = ob_json
        connection_ob = add_connection_to_reply(opt_reply)
        uuuid_x = str(uuid.uuid4())
        connection_ob.connection_id.connection_uuid.uuid = uuuid_x
        connection_ob.service_id.CopyFrom(service.service_id)
       
        ob_id = ob["optical_band_id"]        
        obt = ob["band_type"]
        if obt == "l_slots":
          band_type = "L_BAND"
        elif obt == "s_slots":
          band_type = "S_BAND"
        else:
          band_type = "C_BAND"
          
        freq = ob["freq"]
        bx = ob["band"]
        lf = int(int(freq)-int(bx/2))
        uf = int(int(freq)+int(bx/2))
        val_ob = {"band_type": band_type, "low-freq": lf, "up-freq": uf, "frequency": freq, "band": bx, "ob_id": ob_id}
        rules_ob.append(ConfigRule_Custom(resource_key="/settings-ob_{}".format(uuuid_x), resource_value=json.dumps(val_ob)))
        bidir_ob = ob["bidir"]
        for devxb in ob["flows"].keys():
            log.debug("optical-band device {}".format(devxb))
            in_end_point_b = "0"
            out_end_point_b = "0"
            in_end_point_f = ob["flows"][devxb]["f"]["in"]
            out_end_point_f = ob["flows"][devxb]["f"]["out"]
            log.debug("optical-band ports {}, {}".format(in_end_point_f, out_end_point_f))
            if bidir_ob:
                in_end_point_b = ob["flows"][devxb]["b"]["in"]
                out_end_point_b = ob["flows"][devxb]["b"]["out"]
                log.debug("optical-band ports {}, {}".format(in_end_point_b, out_end_point_b))
            #if (in_end_point_f == "0" or out_end_point_f == "0") and (in_end_point_b == "0" or out_end_point_b == "0"):
            if in_end_point_f != "0":
                d_ob, p_ob = get_uuids_from_names(devices, devxb, in_end_point_f)
                if d_ob != "" and p_ob != "":
                    end_point_b = EndPointId(topology_id=topo, device_id=DeviceId(device_uuid=Uuid(uuid=d_ob)), endpoint_uuid=Uuid(uuid=p_ob))
                    connection_ob.path_hops_endpoint_ids.add().CopyFrom(end_point_b)
                else:
                    log.info("no map device port for device {} port {}".format(devxb, in_end_point_f))

            if out_end_point_f != "0":
                d_ob, p_ob = get_uuids_from_names(devices, devxb, out_end_point_f)
                if d_ob != "" and p_ob != "":
                    end_point_b = EndPointId(topology_id=topo, device_id=DeviceId(device_uuid=Uuid(uuid=d_ob)), endpoint_uuid=Uuid(uuid=p_ob))
                    connection_ob.path_hops_endpoint_ids.add().CopyFrom(end_point_b) 
                else:
                    log.info("no map device port for device {} port {}".format(devxb, out_end_point_f))
            if in_end_point_b != "0":
                d_ob, p_ob = get_uuids_from_names(devices, devxb, in_end_point_b)
                if d_ob != "" and p_ob != "":
                    end_point_b = EndPointId(topology_id=topo, device_id=DeviceId(device_uuid=Uuid(uuid=d_ob)), endpoint_uuid=Uuid(uuid=p_ob))
                    connection_ob.path_hops_endpoint_ids.add().CopyFrom(end_point_b) 
                else:
                    log.info("no map device port for device {} port {}".format(devxb, in_end_point_b))
            if out_end_point_b != "0":
                d_ob, p_ob = get_uuids_from_names(devices, devxb, out_end_point_b)
                if d_ob != "" and p_ob != "":
                    end_point_b = EndPointId(topology_id=topo, device_id=DeviceId(device_uuid=Uuid(uuid=d_ob)), endpoint_uuid=Uuid(uuid=p_ob))
                    connection_ob.path_hops_endpoint_ids.add().CopyFrom(end_point_b)
                else:
                    log.info("no map device port for device {} port {}".format(devxb, out_end_point_b))
            log.debug("optical-band connection {}".format(connection_ob))
    r = reply_json
    bidir_f = r["bidir"]
    connection_f = add_connection_to_reply(opt_reply)
    connection_f.connection_id.connection_uuid.uuid = str(uuid.uuid4())
    connection_f.service_id.CopyFrom(service.service_id)
    for devx in r["flows"].keys():
        log.debug("lightpath device {}".format(devx))
        in_end_point_b = "0"
        out_end_point_b = "0"
        
        in_end_point_f = r["flows"][devx]["f"]["in"]
        out_end_point_f = r["flows"][devx]["f"]["out"]
        log.debug("lightpath ports {}, {}".format(in_end_point_f, out_end_point_f))
        if bidir_f:
            in_end_point_b = r["flows"][devx]["b"]["in"]
            out_end_point_b = r["flows"][devx]["b"]["out"]
            log.debug("lightpath ports {}, {}".format(in_end_point_b, out_end_point_b))
        if in_end_point_f != "0":
            d, p = get_uuids_from_names(devices, devx, in_end_point_f)
            if d != "" and p != "":
                end_point = EndPointId(topology_id=topo, device_id=DeviceId(device_uuid=Uuid(uuid=d)), endpoint_uuid=Uuid(uuid=p))
                connection_f.path_hops_endpoint_ids.add().CopyFrom(end_point) 
            else:
                log.info("no map device port for device {} port {}".format(devx, in_end_point_f))
        if out_end_point_f != "0":
            d, p = get_uuids_from_names(devices, devx, out_end_point_f)
            if d != "" and p != "":
                end_point = EndPointId(topology_id=topo, device_id=DeviceId(device_uuid=Uuid(uuid=d)), endpoint_uuid=Uuid(uuid=p))
                connection_f.path_hops_endpoint_ids.add().CopyFrom(end_point)
            else:
                log.info("no map device port for device {} port {}".format(devx, out_end_point_f))
        if in_end_point_b != "0":
            d, p = get_uuids_from_names(devices, devx, in_end_point_b)
            if d != "" and p != "":
                end_point = EndPointId(topology_id=topo, device_id=DeviceId(device_uuid=Uuid(uuid=d)), endpoint_uuid=Uuid(uuid=p))
                connection_f.path_hops_endpoint_ids.add().CopyFrom(end_point) 
            else:
                log.info("no map device port for device {} port {}".format(devx, in_end_point_b))
        if out_end_point_b != "0":
            d, p = get_uuids_from_names(devices, devx, out_end_point_b)
            if d != "" and p != "":
                end_point = EndPointId(topology_id=topo, device_id=DeviceId(device_uuid=Uuid(uuid=d)), endpoint_uuid=Uuid(uuid=p))
                connection_f.path_hops_endpoint_ids.add().CopyFrom(end_point) 
            else:
                log.info("no map device port for device {} port {}".format(devx, out_end_point_b))      
    #check that list of endpoints is not empty  
    if connection_ob is not None and  len(connection_ob.path_hops_endpoint_ids) == 0:
        log.debug("deleting empty optical-band connection")
        opt_reply.connections.remove(connection_ob)
    
    #inizialize custom optical parameters
    band = r["band"] if "band" in r else None
    op_mode = r["op-mode"] if "op-mode" in r else None
    frequency = r["freq"] if "freq" in r else None
    flow_id = r["flow_id"] if "flow_id" in r else None
    r_type = r["band_type"] if "band_type" in r else None
    if r_type == "l_slots":
        band_type = "L_BAND"
    elif r_type == "s_slots":
        band_type = "S_BAND"
    else:
        band_type = "C_BAND"
        
    if ob_id != 0:
        val = {"frequency": frequency, "operational-mode": op_mode, "band": band, "flow_id": flow_id, "ob_id": ob_id, "band_type": band_type,}
    else:
        val = {"frequency": frequency, "operational-mode": op_mode, "band": band, "flow_id": flow_id, "band_type": band_type,}
    custom_rule = ConfigRule_Custom(resource_key="/settings", resource_value=json.dumps(val))
    rule = ConfigRule(action=ConfigActionEnum.CONFIGACTION_SET, custom=custom_rule)
    service.service_config.config_rules.add().CopyFrom(rule)
   
    if len(rules_ob) > 0:
      for rulex in rules_ob:
        rule_ob = ConfigRule(action=ConfigActionEnum.CONFIGACTION_SET, custom=rulex)
        service.service_config.config_rules.add().CopyFrom(rule_ob)
      
    opt_reply.services.add().CopyFrom(service)   
    
    return opt_reply         

def add_service_to_reply(reply : PathCompReply, service : Service)-> Service:
    service_x = reply.services.add()
    service_x.CopyFrom(service)
    return service_x
 
def add_connection_to_reply(reply : PathCompReply)-> Connection:
    conn = reply.connections.add()
    return conn
