package eu.teraflow.automation;

import eu.teraflow.automation.context.ContextService;
import io.quarkus.runtime.StartupEvent;
import java.time.Duration;
import javax.enterprise.context.ApplicationScoped;
import javax.enterprise.event.Observes;
import javax.inject.Inject;
import org.jboss.logging.Logger;

@ApplicationScoped
public class ContextSubscriber {

    private static final Logger LOGGER = Logger.getLogger(ContextSubscriber.class);

    private final ContextService contextService;
    private final AutomationService automationService;
    private final AutomationConfiguration automationConfiguration;

    @Inject
    public ContextSubscriber(
            ContextService contextService,
            AutomationService automationService,
            AutomationConfiguration automationConfiguration) {
        this.contextService = contextService;
        this.automationService = automationService;
        this.automationConfiguration = automationConfiguration;
    }

    public void listenForDeviceEvents() {

        contextService
                .getDeviceEvents()
                .onFailure()
                .retry()
                .withBackOff(Duration.ofSeconds(1))
                .withJitter(0.2)
                .atMost(10)
                .onFailure()
                .recoverWithCompletion()
                .subscribe()
                .with(
                        deviceEvent -> {
                            LOGGER.debugf("Received %s via contextService:getDeviceEvents", deviceEvent);
                            if (deviceEvent == null || deviceEvent.getEvent() == null) {
                                LOGGER.warn("Received device event is null, ignoring...");
                                return;
                            }
                            final var eventType = deviceEvent.getEvent().getEventTypeEnum();
                            final var deviceId = deviceEvent.getDeviceId();
                            final var event = deviceEvent.getEvent();

                            switch (eventType) {
                                case CREATE:
                                    LOGGER.infof("Received %s for device [%s]", event, deviceId);
                                    automationService.addDevice(deviceEvent.getDeviceId());
                                    break;

                                case UPDATE:
                                case REMOVE:
                                case UNDEFINED:
                                    {
                                        LOGGER.warnf(
                                                "Received %s for device [%s]. [%s] event handling is not yet implemented, ignoring...",
                                                event, deviceId, eventType);
                                    }
                                    break;
                            }
                        });
    }

    void onStart(@Observes StartupEvent ev) {

        if (automationConfiguration.shouldSubscribeToContextComponent()) {
            LOGGER.info("Subscribing to Context service for device events...");
            listenForDeviceEvents();
        } else {
            LOGGER.info("Not subscribing to Context service for device events...");
        }
    }
}
