from __future__ import annotations
from typing import TYPE_CHECKING, Dict
from ..entity._Entity import _Entity
from ..entity.EntityAttributes import EntityAttributes
from .Endpoint import Endpoint
from .Keys import KEY_LINK_ENDPOINT

if TYPE_CHECKING:
    from .Context import Context
    from .Link import Link

VALIDATORS = {
    'device_uuid': lambda v: v is None or isinstance(v, str),
    'endpoint_uuid': lambda v: v is None or isinstance(v, str),
}

class LinkEndpoint(_Entity):
    def __init__(self, link_endpoint_uuid : str, parent : 'Link'):
        super().__init__(parent=parent)
        self._link_endpoint_uuid = link_endpoint_uuid
        self._link_uuid = self._parent.link_uuid
        self._topology_uuid = self._parent.topology_uuid
        self._context_uuid = self._parent.context_uuid
        self._attributes = EntityAttributes(self, KEY_LINK_ENDPOINT, VALIDATORS)

    @property
    def parent(self) -> 'Link': return self._parent

    @property
    def context(self) -> 'Context': return self._parent.context

    @property
    def context_uuid(self) -> str: return self.context.context_uuid

    @property
    def topology_uuid(self) -> str: return self.parent.topology_uuid

    @property
    def link_uuid(self) -> str: return self.parent.link_uuid

    @property
    def link_endpoint_uuid(self) -> str: return self._link_endpoint_uuid

    @property
    def attributes(self) -> EntityAttributes: return self._attributes

    def create(self, endpoint : Endpoint) -> 'LinkEndpoint':
        self.update(update_attributes={
            'device_uuid': endpoint.device_uuid,
            'endpoint_uuid': endpoint.endpoint_uuid,
        })
        self.parent.endpoints.add(self.link_endpoint_uuid)
        return self

    def update(self, update_attributes={}, remove_attributes=[]) -> 'LinkEndpoint':
        self.attributes.update(update_attributes=update_attributes, remove_attributes=remove_attributes)
        return self

    def delete(self) -> None:
        self.attributes.delete()
        self.parent.endpoints.delete(self.link_endpoint_uuid)

    def dump(self) -> Dict:
        attributes = self.attributes.get()
        return {
            'topoId': {'uuid': self.topology_uuid},
            'dev_id': {'uuid': attributes.get('device_uuid', None)},
            'port_id': {'uuid': attributes.get('endpoint_uuid', None)},
        }
