from __future__ import annotations
from typing import TYPE_CHECKING, Dict
from common.database.api.context.topology.device.Endpoint import Endpoint as DeviceEndpoint
from common.database.api.context.Keys import KEY_SERVICE_ENDPOINT
from common.database.api.entity._Entity import _Entity

if TYPE_CHECKING:
    from common.database.api.context.Context import Context
    from common.database.api.context.service.Service import Service

VALIDATORS = {
    'topology_uuid': lambda v: v is not None and isinstance(v, str) and (len(v) > 0),
    'device_uuid': lambda v: v is not None and isinstance(v, str) and (len(v) > 0),
    'endpoint_uuid': lambda v: v is not None and isinstance(v, str) and (len(v) > 0),
}

TRANSCODERS = {} # no transcoding applied to attributes

class Endpoint(_Entity):
    def __init__(self, endpoint_uuid : str, parent : 'Service'):
        super().__init__(parent, endpoint_uuid, KEY_SERVICE_ENDPOINT, VALIDATORS, TRANSCODERS)

    @property
    def parent(self) -> 'Service': return self._parent

    @property
    def context(self) -> 'Context': return self.parent.context

    @property
    def context_uuid(self) -> str: return self.parent.context_uuid

    @property
    def service(self) -> 'Service': return self.parent

    @property
    def service_uuid(self) -> str: return self.parent.service_uuid

    @property
    def endpoint_uuid(self) -> str: return self._entity_uuid

    def create(self, endpoint : DeviceEndpoint) -> 'Endpoint':
        self.update(update_attributes={
            'topology_uuid': endpoint.topology_uuid,
            'device_uuid': endpoint.device_uuid,
            'endpoint_uuid': endpoint.endpoint_uuid,
        })
        self.parent.endpoints.add(self.endpoint_uuid)
        return self

    def update(self, update_attributes={}, remove_attributes=[]) -> 'Endpoint':
        self.attributes.update(update_attributes=update_attributes, remove_attributes=remove_attributes)
        return self

    def delete(self) -> None:
        self.attributes.delete()
        self.parent.endpoints.delete(self.endpoint_uuid)

    def dump_id(self) -> Dict:
        attributes = self.attributes.get()
        topology_uuid = attributes.get('topology_uuid', None)
        device_uuid = attributes.get('device_uuid', None)
        endpoint_uuid = attributes.get('endpoint_uuid', None)
        endpoint = self.context.topology(topology_uuid).device(device_uuid).endpoint(endpoint_uuid)
        return endpoint.dump_id()

    def dump(self) -> Dict:
        return self.dump_id()
