package eu.teraflow.automation;

import automation.Automation;
import context.ContextOuterClass;
import eu.teraflow.automation.device.model.Device;
import eu.teraflow.automation.model.DeviceRoleId;
import io.quarkus.grpc.GrpcService;
import io.smallrye.mutiny.Uni;
import javax.inject.Inject;

@GrpcService
public class AutomationGatewayImpl implements AutomationGateway {

    private final AutomationService automationService;
    private final Serializer serializer;

    @Inject
    public AutomationGatewayImpl(AutomationService automationService, Serializer serializer) {
        this.automationService = automationService;
        this.serializer = serializer;
    }

    @Override
    public Uni<Automation.DeviceRole> ztpGetDeviceRole(Automation.DeviceRoleId request) {
        return Uni.createFrom()
                .item(() -> Automation.DeviceRole.newBuilder().setDevRoleId(request).build());
    }

    @Override
    public Uni<Automation.DeviceRoleList> ztpGetDeviceRolesByDeviceId(
            ContextOuterClass.DeviceId request) {
        return Uni.createFrom().item(() -> Automation.DeviceRoleList.newBuilder().build());
    }

    @Override
    public Uni<Automation.DeviceRoleState> ztpAdd(Automation.DeviceRole request) {
        final var devRoleId = request.getDevRoleId().getDevRoleId().getUuid();
        final var deviceId = serializer.deserialize(request.getDevRoleId().getDevId());

        return automationService
                .addDevice(deviceId)
                .onItem()
                .transform( device -> transformToDeviceRoleState(device, devRoleId));
    }

    @Override
    public Uni<Automation.DeviceRoleState> ztpUpdate(Automation.DeviceRole request) {
        return Uni.createFrom()
                .item(
                        () ->
                                Automation.DeviceRoleState.newBuilder()
                                        .setDevRoleId(request.getDevRoleId())
                                        .build());
    }

    @Override
    public Uni<Automation.DeviceRoleState> ztpDelete(Automation.DeviceRole request) {
        return Uni.createFrom()
                .item(
                        () ->
                                Automation.DeviceRoleState.newBuilder()
                                        .setDevRoleId(request.getDevRoleId())
                                        .build());
    }

    @Override
    public Uni<Automation.DeviceDeletionResult> ztpDeleteAll(Automation.Empty empty) {
        return Uni.createFrom().item(() -> Automation.DeviceDeletionResult.newBuilder().build());
    }

    // TODO When `DeviceRoleState` domain object will be created, move this method to Serializer class and create related tests
    private Automation.DeviceRoleState transformToDeviceRoleState(Device device, String devRoleId){

        final var deviceRoleId = new DeviceRoleId(devRoleId, device.getDeviceId());
        final var serializeDeviceRoleId = serializer.serialize(deviceRoleId);

        return Automation.DeviceRoleState.newBuilder()
                .setDevRoleId(serializeDeviceRoleId)
                .setDevRoleState(Automation.ZtpDeviceState.ZTP_DEV_STATE_CREATED)
                .build();
    }
}