# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import anytree, json, logging
from typing import Any, List, Optional, Tuple, Union
from common.proto.context_pb2 import ConfigActionEnum, ConfigRule, DeviceId, Service
from common.tools.object_factory.Device import json_device_id
from common.type_checkers.Checkers import chk_length, chk_type
from service.service.service_handler_api._ServiceHandler import _ServiceHandler
from service.service.service_handler_api.AnyTreeTools import TreeNode, delete_subnode, get_subnode, set_subnode_value
from service.service.task_scheduler.TaskExecutor import TaskExecutor
from .ConfigRules import setup_config_rules, teardown_config_rules

LOGGER = logging.getLogger(__name__)

class L2NMEmulatedServiceHandler(_ServiceHandler):
    def __init__(   # pylint: disable=super-init-not-called
        self, service : Service, task_executor : TaskExecutor, **settings
    ) -> None:
        self.__service = service
        self.__task_executor = task_executor # pylint: disable=unused-private-member
        self.__resolver = anytree.Resolver(pathattr='name')
        self.__config = TreeNode('.')
        for config_rule in service.service_config.config_rules:
            action = config_rule.action
            if config_rule.WhichOneof('config_rule') != 'custom': continue
            resource_key = config_rule.custom.resource_key
            resource_value = config_rule.custom.resource_value
            if action == ConfigActionEnum.CONFIGACTION_SET:
                try:
                    resource_value = json.loads(resource_value)
                except: # pylint: disable=bare-except
                    pass
                set_subnode_value(self.__resolver, self.__config, resource_key, resource_value)
            elif action == ConfigActionEnum.CONFIGACTION_DELETE:
                delete_subnode(self.__resolver, self.__config, resource_key)

    def SetEndpoint(
        self, endpoints : List[Tuple[str, str, Optional[str]]], connection_uuid : Optional[str] = None
    ) -> List[Union[bool, Exception]]:
        chk_type('endpoints', endpoints, list)
        if len(endpoints) == 0: return []

        service_uuid = self.__service.service_id.service_uuid.uuid
        settings : TreeNode = get_subnode(self.__resolver, self.__config, '/settings', None)

        results = []
        for endpoint in endpoints:
            try:
                chk_type('endpoint', endpoint, (tuple, list))
                chk_length('endpoint', endpoint, min_length=2, max_length=3)
                device_uuid, endpoint_uuid = endpoint[0:2] # ignore topology_uuid by now

                endpoint_settings_uri = '/device[{:s}]/endpoint[{:s}]/settings'.format(device_uuid, endpoint_uuid)
                endpoint_settings : TreeNode = get_subnode(self.__resolver, self.__config, endpoint_settings_uri, None)

                json_config_rules = setup_config_rules(
                    service_uuid, connection_uuid, device_uuid, endpoint_uuid, settings, endpoint_settings)

                device = self.__task_executor.get_device(DeviceId(**json_device_id(device_uuid)))
                for json_config_rule in json_config_rules:
                    device.device_config.config_rules.append(ConfigRule(**json_config_rule))
                self.__task_executor.configure_device(device)
                results.append(True)
            except Exception as e: # pylint: disable=broad-except
                LOGGER.exception('Unable to SetEndpoint({:s})'.format(str(endpoint)))
                results.append(e)

        return results

    def DeleteEndpoint(
        self, endpoints : List[Tuple[str, str, Optional[str]]], connection_uuid : Optional[str] = None
    ) -> List[Union[bool, Exception]]:
        chk_type('endpoints', endpoints, list)
        if len(endpoints) == 0: return []

        service_uuid = self.__service.service_uuid.uuid
        settings : TreeNode = get_subnode(self.__resolver, self.__config, '/settings', None)

        results = []
        for endpoint in endpoints:
            try:
                chk_type('endpoint', endpoint, (tuple, list))
                chk_length('endpoint', endpoint, min_length=2, max_length=3)
                device_uuid, endpoint_uuid = endpoint[0:2] # ignore topology_uuid by now

                endpoint_settings_uri = '/device[{:s}]/endpoint[{:s}]/settings'.format(device_uuid, endpoint_uuid)
                endpoint_settings : TreeNode = get_subnode(self.__resolver, self.__config, endpoint_settings_uri, None)

                json_config_rules = teardown_config_rules(
                    service_uuid, connection_uuid, device_uuid, endpoint_uuid, settings, endpoint_settings)

                device = self.__task_executor.get_device(DeviceId(**json_device_id(device_uuid)))
                for json_config_rule in json_config_rules:
                    device.device_config.config_rules.append(ConfigRule(**json_config_rule))
                self.__task_executor.configure_device(device)
                results.append(True)
            except Exception as e: # pylint: disable=broad-except
                LOGGER.exception('Unable to DeleteEndpoint({:s})'.format(str(endpoint)))
                results.append(e)

        return results

    def SetConstraint(self, constraints : List[Tuple[str, Any]]) -> List[Union[bool, Exception]]:
        chk_type('constraints', constraints, list)
        if len(constraints) == 0: return []

        msg = '[SetConstraint] Method not implemented. Constraints({:s}) are being ignored.'
        LOGGER.warning(msg.format(str(constraints)))
        return [True for _ in range(len(constraints))]

    def DeleteConstraint(self, constraints : List[Tuple[str, Any]]) -> List[Union[bool, Exception]]:
        chk_type('constraints', constraints, list)
        if len(constraints) == 0: return []

        msg = '[DeleteConstraint] Method not implemented. Constraints({:s}) are being ignored.'
        LOGGER.warning(msg.format(str(constraints)))
        return [True for _ in range(len(constraints))]

    def SetConfig(self, resources : List[Tuple[str, Any]]) -> List[Union[bool, Exception]]:
        chk_type('resources', resources, list)
        if len(resources) == 0: return []

        results = []
        for resource in resources:
            try:
                resource_key, resource_value = resource
                resource_value = json.loads(resource_value)
                set_subnode_value(self.__resolver, self.__config, resource_key, resource_value)
                results.append(True)
            except Exception as e: # pylint: disable=broad-except
                LOGGER.exception('Unable to SetConfig({:s})'.format(str(resource)))
                results.append(e)

        return results

    def DeleteConfig(self, resources : List[Tuple[str, Any]]) -> List[Union[bool, Exception]]:
        chk_type('resources', resources, list)
        if len(resources) == 0: return []

        results = []
        for resource in resources:
            try:
                resource_key, _ = resource
                delete_subnode(self.__resolver, self.__config, resource_key)
            except Exception as e: # pylint: disable=broad-except
                LOGGER.exception('Unable to DeleteConfig({:s})'.format(str(resource)))
                results.append(e)

        return results
