# Copyright 2022-2023 ETSI TeraFlowSDN - TFS OSG (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from flask import render_template, Blueprint, flash
from common.proto.context_pb2 import Empty
from common.proto.load_generator_pb2 import Parameters, RequestTypeEnum
from load_generator.client.LoadGeneratorClient import LoadGeneratorClient
from .forms import LoadGenStartForm, LoadGenStopForm

load_gen = Blueprint('load_gen', __name__, url_prefix='/load_gen')

@load_gen.route('home', methods=['GET', 'POST'])
def home():
    load_gen_client = LoadGeneratorClient()

    form_start = LoadGenStartForm()
    form_stop = LoadGenStopForm()

    if form_start.validate_on_submit():
        load_gen_params = Parameters()
        load_gen_params.num_requests       = form_start.num_requests.data
        load_gen_params.offered_load       = form_start.offered_load.data
        load_gen_params.holding_time       = form_start.holding_time.data
        load_gen_params.inter_arrival_time = form_start.inter_arrival_time.data
        load_gen_params.do_teardown        = form_start.do_teardown.data
        load_gen_params.dry_mode           = False
        load_gen_params.record_to_dlt      = form_start.record_to_dlt.data
        load_gen_params.dlt_domain_id      = form_start.dlt_domain_id.data

        del load_gen_params.request_types[:] # pylint: disable=no-member
        request_types = list()
        if form_start.request_type_service_l2nm.data: request_types.append(RequestTypeEnum.REQUESTTYPE_SERVICE_L2NM)
        if form_start.request_type_service_l3nm.data: request_types.append(RequestTypeEnum.REQUESTTYPE_SERVICE_L3NM)
        if form_start.request_type_service_mw  .data: request_types.append(RequestTypeEnum.REQUESTTYPE_SERVICE_MW  )
        if form_start.request_type_service_tapi.data: request_types.append(RequestTypeEnum.REQUESTTYPE_SERVICE_TAPI)
        if form_start.request_type_slice_l2nm  .data: request_types.append(RequestTypeEnum.REQUESTTYPE_SLICE_L2NM  )
        if form_start.request_type_slice_l3nm  .data: request_types.append(RequestTypeEnum.REQUESTTYPE_SLICE_L3NM  )
        load_gen_params.request_types.extend(request_types) # pylint: disable=no-member

        try:
            load_gen_client.connect()
            load_gen_client.Start(load_gen_params)
            load_gen_client.close()
            flash('Load Generator Started.', 'success')
        except Exception as e: # pylint: disable=broad-except
            flash('Problem starting Load Generator. {:s}'.format(str(e)), 'danger')

    if form_stop.validate_on_submit():
        try:
            load_gen_client.connect()
            load_gen_client.Stop(Empty())
            load_gen_client.close()
            flash('Load Generator Stoped.', 'success')
        except Exception as e: # pylint: disable=broad-except
            flash('Problem stopping Load Generator. {:s}'.format(str(e)), 'danger')

    load_gen_client.connect()
    status = load_gen_client.GetStatus(Empty())
    load_gen_client.close()

    form_start.num_requests      .default = status.parameters.num_requests
    form_start.offered_load      .default = status.parameters.offered_load
    form_start.holding_time      .default = status.parameters.holding_time
    form_start.inter_arrival_time.default = status.parameters.inter_arrival_time
    form_start.do_teardown       .default = status.parameters.do_teardown
    form_start.record_to_dlt     .default = status.parameters.record_to_dlt
    form_start.dlt_domain_id     .default = status.parameters.dlt_domain_id
    form_start.num_generated     .default = status.num_generated
    form_start.infinite_loop     .default = status.infinite_loop
    form_start.running           .default = status.running

    request_types = status.parameters.request_types
    form_start.request_type_service_l2nm.default = RequestTypeEnum.REQUESTTYPE_SERVICE_L2NM in request_types
    form_start.request_type_service_l3nm.default = RequestTypeEnum.REQUESTTYPE_SERVICE_L3NM in request_types
    form_start.request_type_service_mw  .default = RequestTypeEnum.REQUESTTYPE_SERVICE_MW   in request_types
    form_start.request_type_service_tapi.default = RequestTypeEnum.REQUESTTYPE_SERVICE_TAPI in request_types
    form_start.request_type_slice_l2nm  .default = RequestTypeEnum.REQUESTTYPE_SLICE_L2NM   in request_types
    form_start.request_type_slice_l3nm  .default = RequestTypeEnum.REQUESTTYPE_SLICE_L3NM   in request_types

    form_start.num_requests      .render_kw['readonly'] = status.running
    form_start.offered_load      .render_kw['readonly'] = status.running
    form_start.holding_time      .render_kw['readonly'] = status.running
    form_start.inter_arrival_time.render_kw['readonly'] = status.running
    form_start.do_teardown       .render_kw['readonly'] = status.running
    form_start.record_to_dlt     .render_kw['readonly'] = status.running
    form_start.dlt_domain_id     .render_kw['readonly'] = status.running

    form_start.request_type_service_l2nm.render_kw['readonly'] = status.running
    form_start.request_type_service_l3nm.render_kw['readonly'] = status.running
    form_start.request_type_service_mw  .render_kw['readonly'] = status.running
    form_start.request_type_service_tapi.render_kw['readonly'] = status.running
    form_start.request_type_slice_l2nm  .render_kw['readonly'] = status.running
    form_start.request_type_slice_l3nm  .render_kw['readonly'] = status.running

    return render_template('load_gen/home.html', form_start=form_start, form_stop=form_stop)
