import logging, signal, sys, threading
from prometheus_client import start_http_server
from common.Settings import get_setting
from slice.client.SliceClient import SliceClient

from interdomain.Config import (
    SLICE_SERVICE_HOST, SLICE_SERVICE_PORT, GRPC_INTERDOMAIN_PORT,
    GRPC_MAX_WORKERS, GRPC_GRACE_PERIOD, LOG_LEVEL, METRICS_PORT)
from .InterdomainService import InterdomainService


terminate = threading.Event()
LOGGER = None

def signal_handler(signal, frame): # pylint: disable=redefined-outer-name
    LOGGER.warning('Terminate signal received')
    terminate.set()

def main():
    global LOGGER # pylint: disable=global-statement

    grpc_service_port    = get_setting('INTERDOMAINSERVICE_SERVICE_PORT_GRPC', default=GRPC_INTERDOMAIN_PORT   )
    max_workers          = get_setting('MAX_WORKERS',                      default=GRPC_MAX_WORKERS    )
    grace_period         = get_setting('GRACE_PERIOD',                     default=GRPC_GRACE_PERIOD   )
    log_level            = get_setting('LOG_LEVEL',                        default=LOG_LEVEL           )
    metrics_port         = get_setting('METRICS_PORT',                     default=METRICS_PORT        )
    slice_service_host = get_setting('SLICESERVICE_SERVICE_HOST',      default=SLICE_SERVICE_HOST)
    slice_service_port = get_setting('SLICESERVICE_SERVICE_PORT_GRPC', default=SLICE_SERVICE_PORT)


    logging.basicConfig(level=log_level)
    LOGGER = logging.getLogger(__name__)

    signal.signal(signal.SIGINT,  signal_handler)
    signal.signal(signal.SIGTERM, signal_handler)

    LOGGER.info('Starting...')

    # Start metrics server
    start_http_server(metrics_port)

    # Initialize Slice Client
    if slice_service_host is None or slice_service_port is None:
        raise Exception('Wrong address({:s}):port({:s}) of Device component'.format(
            str(slice_service_host), str(slice_service_port)))
    slice_client = SliceClient(slice_service_host, slice_service_port)

    # Starting service service
    grpc_interdomain = InterdomainService( slice_client=slice_client, port=grpc_interdomain_port, max_workers=max_workers, grace_period=grace_period)
    grpc_interdomain.start()

    # Wait for Ctrl+C or termination signal
    while not terminate.wait(timeout=0.1): pass

    LOGGER.info('Terminating...')
    grpc_interdomain.stop()

    LOGGER.info('Bye')
    return 0

if __name__ == '__main__':
    sys.exit(main())
