#!/bin/bash
# Copyright 2022-2024 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

set -euo pipefail

# -----------------------------------------------------------
# Global namespace for all deployments
# -----------------------------------------------------------
NAMESPACE="monitoring"
VALUES_FILE_PATH="manifests/monitoring"

# -----------------------------------------------------------
# Prometheus Configuration
# -----------------------------------------------------------
RELEASE_NAME_PROM="mon-prometheus"
CHART_REPO_NAME_PROM="prometheus-community"
CHART_REPO_URL_PROM="https://prometheus-community.github.io/helm-charts"
CHART_NAME_PROM="prometheus"
VALUES_FILE_PROM="$VALUES_FILE_PATH/prometheus_values.yaml"

# -----------------------------------------------------------
# Mimir Configuration
# -----------------------------------------------------------
# RELEASE_NAME_MIMIR="mon-mimir"
# CHART_REPO_NAME_MIMIR="grafana"
# CHART_REPO_URL_MIMIR="https://grafana.github.io/helm-charts"
# CHART_NAME_MIMIR="mimir-distributed"
# VALUES_FILE_MIMIR="$VALUES_FILE_PATH/mimir_values.yaml"

# -----------------------------------------------------------
# Grafana Configuration
# -----------------------------------------------------------
# RELEASE_NAME_GRAFANA="mon-grafana"
# CHART_REPO_NAME_GRAFANA="grafana"
# CHART_REPO_URL_GRAFANA="https://grafana.github.io/helm-charts"
# CHART_NAME_GRAFANA="grafana"
# VALUES_FILE_GRAFANA="$VALUES_FILE_PATH/grafana_values.yaml"


# -----------------------------------------------------------
# Function to deploy or upgrade a Helm chart
# -----------------------------------------------------------
deploy_chart() {
  local release_name="$1"
  local chart_repo_name="$2"
  local chart_repo_url="$3"
  local chart_name="$4"
  local values_file="$5"
  local namespace="$6"

  echo ">>> Deploying [${release_name}] from repo [${chart_repo_name}]..."

  # Add or update the Helm repo
  echo "Adding/updating Helm repo: $chart_repo_name -> $chart_repo_url"
  helm repo add "$chart_repo_name" "$chart_repo_url" || true
  helm repo update

  # Create namespace if needed
  echo "Creating namespace '$namespace' if it doesn't exist..."
  kubectl get namespace "$namespace" >/dev/null 2>&1 || kubectl create namespace "$namespace"

  # Install or upgrade the chart
  if [ -n "$values_file" ] && [ -f "$values_file" ]; then
    echo "Installing/Upgrading $release_name using custom values from $values_file..."
    helm upgrade --install "$release_name" "$chart_repo_name/$chart_name" \
      --namespace "$namespace" \
      --values "$values_file"
  else
    echo "Installing/Upgrading $release_name with default chart values..."
    helm upgrade --install "$release_name" "$chart_repo_name/$chart_name" \
      --namespace "$namespace"
  fi

  echo "<<< Deployment initiated for [$release_name]."
  echo
}


# -----------------------------------------------------------
# Actual Deployments
# -----------------------------------------------------------

# 1) Deploy Prometheus
deploy_chart "$RELEASE_NAME_PROM" \
             "$CHART_REPO_NAME_PROM" \
             "$CHART_REPO_URL_PROM" \
             "$CHART_NAME_PROM" \
             "$VALUES_FILE_PROM" \
             "$NAMESPACE"

# Optionally wait for Prometheus server pod to become ready
kubectl rollout status deployment/"$RELEASE_NAME_PROM-server" -n "$NAMESPACE" || true


# 2) Deploy Mimir
# deploy_chart "$RELEASE_NAME_MIMIR" \
#              "$CHART_REPO_NAME_MIMIR" \
#              "$CHART_REPO_URL_MIMIR" \
#              "$CHART_NAME_MIMIR" \
#              "$VALUES_FILE_MIMIR" \
#              "$NAMESPACE"

# Depending on how Mimir runs (StatefulSets, Deployments), you can wait for
# the correct resource to be ready. For example:
# kubectl rollout status statefulset/"$RELEASE_NAME_MIMIR-distributor" -n "$NAMESPACE" || true


# 3) Deploy Grafana
# deploy_chart "$RELEASE_NAME_GRAFANA" \
#              "$CHART_REPO_NAME_GRAFANA" \
#              "$CHART_REPO_URL_GRAFANA" \
#              "$CHART_NAME_GRAFANA" \
#              "$VALUES_FILE_GRAFANA" \
#              "$NAMESPACE"

# kubectl rollout status deployment/"$RELEASE_NAME_GRAFANA" -n "$NAMESPACE" || true

# -----------------------------------------------------------
echo "All deployments completed!"

