# Copyright 2022-2024 ETSI OSG/SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import json, logging
from sqlalchemy.dialects.postgresql import insert
from sqlalchemy.engine import Engine
from sqlalchemy.orm import Session, sessionmaker
from sqlalchemy_cockroachdb import run_transaction
from common.proto.context_pb2 import OpticalConfig, OpticalConfigId
from .models.OpticalConfigModel import OpticalConfigModel

LOGGER = logging.getLogger(__name__)

def get_opticalconfig(db_engine : Engine):
    def callback(session:Session):
        optical_configs = list()
        results = session.query(OpticalConfigModel).all()
        for obj in results:
            optical_config = OpticalConfig()
            optical_config.config = json.dump(obj.config)
            optical_config.opticalconfig_id.opticalconfig_uuid = obj.opticalconfig_uuid
            optical_configs.append(optical_config)
        return optical_configs
    obj = run_transaction(sessionmaker(bind=db_engine), callback)
    return obj

def set_opticalconfig(db_engine : Engine, request : OpticalConfig):
    opticalconfig_id = OpticalConfigId()
    opticalconfig_id.opticalconfig_uuid = request.opticalconfig_id.opticalconfig_uuid
    my_config_data = []
    if request.config:
        channels = []
        transceivers = []
        config = json.loads(request.config)
        if 'channels' in config and len(config['channels']) > 0:
            channels = [channel['name']['index'] for channel in config['channels']]
        if 'transceivers' in config and len(config['transceivers']['transceiver']) > 0:
            transceivers = [transceiver for transceiver in config['transceivers']['transceiver']]

        my_config_data = [
            {
                "opticalconfig_uuid": request.opticalconfig_id.opticalconfig_uuid,
                "channels"          : channels,
                "transcievers"      : transceivers,
                "interfaces"        : json.dumps(config["interfaces"]["interface"]),
                "channel_namespace" : config["channel_namespace"],
                "endpoints"         : [json.dumps(endpoint) for endpoint in config["endpoints"]],
                "frequency"         : config["frequency"] if "frequency" in config  else 0,
                "operational_mode"  : config["operational_mode"] if "operational_mode" in config else 0,
                "output_power"      : config["output_power"] if "output_power" in config else '',
            }
        ]

    def callback(session:Session)->bool:
        stmt = insert(OpticalConfigModel).values(my_config_data)
        stmt = stmt.on_conflict_do_update(
            index_elements=[OpticalConfigModel.opticalconfig_uuid],
            set_=dict(
                channel_namespace=stmt.excluded.channel_namespace
            )
        )
        stmt = stmt.returning(OpticalConfigModel.opticalconfig_uuid)
        id = session.execute(stmt).fetchone()
    opticalconfig_id = run_transaction(sessionmaker(bind=db_engine), callback)
    return {'opticalconfig_uuid': opticalconfig_id}

def select_opticalconfig(db_engine:Engine,request:OpticalConfigId):
    def callback(session : Session) -> OpticalConfig:
        result = OpticalConfig()
        stmt = session.query(OpticalConfigModel)
        stmt = stmt.filter_by(opticalconfig_uuid=request.opticalconfig_uuid)
        obj = stmt.first()
        if obj is not None:
            result.config = json.dumps(obj.dump())
            result.opticalconfig_id.opticalconfig_uuid = obj.opticalconfig_uuid
        return result
    return run_transaction(sessionmaker(bind=db_engine, expire_on_commit=False), callback)
