import logging, os, signal, sys, threading
from prometheus_client import start_http_server
from common.database.Factory import get_database
from slice.service.SliceService import SliceService
from slice.Config import GRPC_SERVICE_PORT, GRPC_MAX_WORKERS, GRPC_GRACE_PERIOD, LOG_LEVEL, METRICS_PORT

terminate = threading.Event()
logger = None

def signal_handler(signal, frame):
    global terminate, logger
    logger.warning('Terminate signal received')
    terminate.set()

def main():
    global terminate, logger

    service_port = os.environ.get('SLICESERVICE_SERVICE_PORT_GRPC', GRPC_SERVICE_PORT)
    max_workers  = os.environ.get('MAX_WORKERS',                    GRPC_MAX_WORKERS )
    grace_period = os.environ.get('GRACE_PERIOD',                   GRPC_GRACE_PERIOD)
    log_level    = os.environ.get('LOG_LEVEL',                      LOG_LEVEL        )
    metrics_port = os.environ.get('METRICS_PORT',                   METRICS_PORT     )

    logging.basicConfig(level=log_level)
    logger = logging.getLogger(__name__)

    signal.signal(signal.SIGINT,  signal_handler)
    signal.signal(signal.SIGTERM, signal_handler)

    logger.info('Starting...')

    # Start metrics server
    start_http_server(metrics_port)

    # Get database instance
    database = get_database()

    # Starting device service
    grpc_service = SliceService(database, port=service_port, max_workers=max_workers, grace_period=grace_period)
    grpc_service.start()

    # Wait for Ctrl+C or termination signal
    while not terminate.wait(timeout=0.1): pass

    logger.info('Terminating...')
    grpc_service.stop()

    logger.info('Bye')
    return 0

if __name__ == '__main__':
    sys.exit(main())
