import logging
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy import Column, Integer, String, Text, Boolean
from sqlalchemy.orm import registry
from common.proto.pcep_pb2 import LspDescriptor
import uuid

# Configuración del logger
logging.basicConfig(level=logging.INFO)
LOGGER = logging.getLogger(__name__)

# Base para los modelos declarativos
Base = registry().generate_base()

class LspModel(Base):
    __tablename__ = 'lsp'

    lsp_id                = Column(UUID(as_uuid=True), primary_key=True)  # Cambiado a as_uuid=True
    delegate_flag         = Column(Boolean, nullable=False)
    sync_flag             = Column(Boolean, nullable=False)
    remove_flag           = Column(Boolean, nullable=False)
    administrative_flag   = Column(Boolean, nullable=False)
    op_flags              = Column(Integer, nullable=False)
    create_flag           = Column(Boolean, nullable=False)
    p2mp_flag             = Column(Boolean, nullable=False)
    fragmentation_flag    = Column(Boolean, nullable=False)
    ero_compression_flag = Column(Boolean, nullable=False)
    symbolic_path_name_tlv = Column(Text, nullable=True)
    lsp_identifiers_tlv    = Column(Text, nullable=True)
    lsp_error_codes_tlv    = Column(Text, nullable=True)
    rsvp_error_spec_tlv    = Column(Text, nullable=True)
    lsp_db_version_tlv     = Column(Text, nullable=True)

    def __repr__(self):
        return (f"<Lsp(lsp_id='{self.lsp_id}', delegate_flag='{self.delegate_flag}', "
                f"sync_flag='{self.sync_flag}', remove_flag='{self.remove_flag}', "
                f"administrative_flag='{self.administrative_flag}', op_flags='{self.op_flags}', "
                f"create_flag='{self.create_flag}', p2mp_flag='{self.p2mp_flag}', "
                f"fragmentation_flag='{self.fragmentation_flag}', ero_compression_flag='{self.ero_compression_flag}', "
                f"symbolic_path_name_tlv='{self.symbolic_path_name_tlv}', lsp_identifiers_tlv='{self.lsp_identifiers_tlv}', "
                f"lsp_error_codes_tlv='{self.lsp_error_codes_tlv}', rsvp_error_spec_tlv='{self.rsvp_error_spec_tlv}', "
                f"lsp_db_version_tlv='{self.lsp_db_version_tlv}')>")

    @classmethod
    def convert_LspDescriptor_to_row(cls, request):
        """
        Create an instance of Lsp from a request object.
        
        Args:   
            request: The request object containing the data.
            
        Returns: 
            An instance of Lsp initialized with data from the request.
        """
        return cls(
            lsp_id                = uuid.UUID(request.lsp_id.lsp_id.uuid),  # Convertir a UUID de la request
            delegate_flag         = request.delegate_flag,
            sync_flag             = request.sync_flag,
            remove_flag           = request.remove_flag,
            administrative_flag   = request.administrative_flag,
            op_flags              = request.op_flags,
            create_flag           = request.create_flag,
            p2mp_flag             = request.p2mp_flag,
            fragmentation_flag    = request.fragmentation_flag,
            ero_compression_flag = request.ero_compression_flag,
            symbolic_path_name_tlv = request.symbolic_path_name_tlv,
            lsp_identifiers_tlv    = request.lsp_identifiers_tlv,
            lsp_error_codes_tlv    = request.lsp_error_codes_tlv,
            rsvp_error_spec_tlv    = request.rsvp_error_spec_tlv,
            lsp_db_version_tlv     = request.lsp_db_version_tlv
        )
    
    @classmethod
    def convert_row_to_LspDescriptor(cls, row):
        """
        Create and return an LspDescriptor object from a Lsp instance.
        
        Args:   
            row: The Lsp instance (row) containing the data.
            
        Returns: 
            LspDescriptor object
        """
        response = LspDescriptor()
        response.lsp_id = str(row.lsp_id)  # Si lsp_id es UUID, convertir a string
        response.delegate_flag = row.delegate_flag
        response.sync_flag = row.sync_flag
        response.remove_flag = row.remove_flag
        response.administrative_flag = row.administrative_flag
        response.op_flags = row.op_flags
        response.create_flag = row.create_flag
        response.p2mp_flag = row.p2mp_flag
        response.fragmentation_flag = row.fragmentation_flag
        response.ero_compression_flag = row.ero_compression_flag
        response.symbolic_path_name_tlv = row.symbolic_path_name_tlv
        response.lsp_identifiers_tlv = row.lsp_identifiers_tlv
        response.lsp_error_codes_tlv = row.lsp_error_codes_tlv
        response.rsvp_error_spec_tlv = row.rsvp_error_spec_tlv
        response.lsp_db_version_tlv = row.lsp_db_version_tlv
        return response
