import grpc, logging
from common.tools.client.RetryDecorator import retry, delay_exponential
from compute.proto.compute_pb2_grpc import ComputeServiceStub
from compute.proto.context_pb2 import AuthenticationResult, Empty, TeraFlowController
from compute.proto.service_pb2 import Service, ServiceId, ServiceIdList, ServiceState

LOGGER = logging.getLogger(__name__)
MAX_RETRIES = 15
DELAY_FUNCTION = delay_exponential(initial=0.01, increment=2.0, maximum=5.0)

class ComputeClient:
    def __init__(self, address, port):
        self.endpoint = '{}:{}'.format(address, port)
        LOGGER.debug('Creating channel to {}...'.format(self.endpoint))
        self.channel = None
        self.stub = None
        self.connect()
        LOGGER.debug('Channel created')

    def connect(self):
        self.channel = grpc.insecure_channel(self.endpoint)
        self.stub = ComputeServiceStub(self.channel)

    def close(self):
        if(self.channel is not None): self.channel.close()
        self.channel = None
        self.stub = None

    @retry(exceptions=set(), max_retries=MAX_RETRIES, delay_function=DELAY_FUNCTION, prepare_method_name='connect')
    def CheckCredentials(self, request : TeraFlowController) -> AuthenticationResult:
        LOGGER.debug('CheckCredentials request: {}'.format(request))
        response = self.stub.check_credentials(request)
        LOGGER.debug('CheckCredentials result: {}'.format(response))
        return response


    @retry(exceptions=set(), max_retries=MAX_RETRIES, delay_function=DELAY_FUNCTION, prepare_method_name='connect')
    def GetConnectivityServiceStatus(self, request : ServiceId) -> ServiceState:
        LOGGER.debug('GetConnectivityServiceStatus request: {}'.format(request))
        response = self.stub.get_connectivity_service_status(request)
        LOGGER.debug('GetConnectivityServiceStatus result: {}'.format(response))
        return response


    @retry(exceptions=set(), max_retries=MAX_RETRIES, delay_function=DELAY_FUNCTION, prepare_method_name='connect')
    def CreateConnectivityService(self, request : Service) -> ServiceId:
        LOGGER.debug('CreateConnectivityService request: {}'.format(request))
        response = self.stub.create_connectivity_service(request)
        LOGGER.debug('CreateConnectivityService result: {}'.format(response))
        return response


    @retry(exceptions=set(), max_retries=MAX_RETRIES, delay_function=DELAY_FUNCTION, prepare_method_name='connect')
    def EditConnectivityService(self, request : Service) -> ServiceId:
        LOGGER.debug('EditConnectivityService request: {}'.format(request))
        response = self.stub.edit_connectivity_service(request)
        LOGGER.debug('EditConnectivityService result: {}'.format(response))
        return response


    @retry(exceptions=set(), max_retries=MAX_RETRIES, delay_function=DELAY_FUNCTION, prepare_method_name='connect')
    def DeleteConnectivityService(self, request : Service) -> Empty:
        LOGGER.debug('DeleteConnectivityService request: {}'.format(request))
        response = self.stub.delete_connectivity_service(request)
        LOGGER.debug('DeleteConnectivityService result: {}'.format(response))
        return response


    @retry(exceptions=set(), max_retries=MAX_RETRIES, delay_function=DELAY_FUNCTION, prepare_method_name='connect')
    def GetAllActiveConnectivityServices(self, request : Empty) -> ServiceIdList:
        LOGGER.debug('GetAllActiveConnectivityServices request: {}'.format(request))
        response = self.stub.get_all_active_connectivity_services(request)
        LOGGER.debug('GetAllActiveConnectivityServices result: {}'.format(response))
        return response


    @retry(exceptions=set(), max_retries=MAX_RETRIES, delay_function=DELAY_FUNCTION, prepare_method_name='connect')
    def ClearAllConnectivityServices(self, request : Empty) -> Empty:
        LOGGER.debug('ClearAllConnectivityServices request: {}'.format(request))
        response = self.stub.clear_all_connectivity_services(request)
        LOGGER.debug('ClearAllConnectivityServices result: {}'.format(response))
        return response
