# Copyright 2022-2024 ETSI OSG/SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import grpc, logging
from typing import Iterator

import grpc._channel
from common.method_wrappers.Decorator import MetricsPool, safe_and_metered_rpc_method
from common.proto.context_pb2 import Empty, QoSProfileId, QoSProfile
from common.proto.qos_profile_pb2_grpc import QoSProfileServiceServicer
from context.client.ContextClient import ContextClient


LOGGER = logging.getLogger(__name__)

METRICS_POOL = MetricsPool('QoSProfile', 'RPC')

class QoSProfileServiceServicerImpl(QoSProfileServiceServicer):
    def __init__(self ) -> None:
        LOGGER.debug('Servicer Created')

    @safe_and_metered_rpc_method(METRICS_POOL, LOGGER)
    def CreateQoSProfile(self, request: QoSProfile, context: grpc.ServicerContext) -> QoSProfile:
        context_client = ContextClient()
        try:
            qos_profile_get = context_client.GetQoSProfile(request.qos_profile_id)
            context.set_details(f'QoSProfile {request.qos_profile_id.qos_profile_id.uuid} already exists')
            context.set_code(grpc.StatusCode.ALREADY_EXISTS)
            return QoSProfile()
        except grpc._channel._InactiveRpcError as exc:
            if exc.code() != grpc.StatusCode.NOT_FOUND:
                raise exc
        qos_profile = context_client.CreateQoSProfile(request)
        return qos_profile

    @safe_and_metered_rpc_method(METRICS_POOL, LOGGER)
    def UpdateQoSProfile(self, request: QoSProfile, context: grpc.ServicerContext) -> QoSProfile:
        context_client = ContextClient()
        try:
            _ = context_client.GetQoSProfile(request.qos_profile_id)
        except grpc._channel._InactiveRpcError as exc:
            if exc.code() == grpc.StatusCode.NOT_FOUND:
                context.set_details(f'QoSProfile {request.qos_profile_id.qos_profile_id.uuid} not found')
                context.set_code(grpc.StatusCode.NOT_FOUND)
                return QoSProfile()
        qos_profile = context_client.UpdateQoSProfile(request)
        return qos_profile

    @safe_and_metered_rpc_method(METRICS_POOL, LOGGER)
    def DeleteQoSProfile(self, request: QoSProfileId, context: grpc.ServicerContext) -> Empty:
        context_client = ContextClient()
        try:
            _ = context_client.GetQoSProfile(request)
        except grpc._channel._InactiveRpcError as exc:
            if exc.code() == grpc.StatusCode.NOT_FOUND:
                context.set_details(f'QoSProfile {request.qos_profile_id.uuid} not found')
                context.set_code(grpc.StatusCode.NOT_FOUND)
                return QoSProfile()
        empty = context_client.DeleteQoSProfile(request)
        return empty

    @safe_and_metered_rpc_method(METRICS_POOL, LOGGER)
    def GetQoSProfile(self, request: QoSProfileId, context: grpc.ServicerContext) -> QoSProfile:
        context_client = ContextClient()
        try:
            qos_profile = context_client.GetQoSProfile(request)
        except grpc._channel._InactiveRpcError as exc:
            if exc.code() == grpc.StatusCode.NOT_FOUND:
                context.set_details(f'QoSProfile {request.qos_profile_id.uuid} not found')
                context.set_code(grpc.StatusCode.NOT_FOUND)
                return QoSProfile()
        return qos_profile

    @safe_and_metered_rpc_method(METRICS_POOL, LOGGER)
    def GetQoSProfiles(self, request: Empty, context: grpc.ServicerContext) -> Iterator[QoSProfile]:
        context_client = ContextClient()
        yield from context_client.GetQoSProfiles(request)

