# Copyright 2022-2023 ETSI TeraFlowSDN - TFS OSG (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from typing import Dict, List, Optional, Tuple
from common.proto.context_pb2 import Device, EndPoint
from common.tools.object_factory.ConfigRule import json_config_rule_delete, json_config_rule_set

from service.service.service_handler_api.AnyTreeTools import TreeNode

def _interface(if_name, sif_index, ipv4_address, ipv4_prefix, enabled) -> Tuple[str, Dict]:
    str_path = '/interface[{:s}]'.format(if_name)
    str_data = {'name': if_name, 'enabled': enabled, 'sub_if_index': sif_index,
                'sub_if_enabled': enabled, 'sub_if_ipv4_enabled': enabled,
                'sub_if_ipv4_address': ipv4_address, 'sub_if_ipv4_prefix': ipv4_prefix}
    return str_path, str_data

def _network_instance(ni_name, ni_type) -> Tuple[str, Dict]:
    str_path = '/network_instance[{:s}]'.format(ni_name)
    str_data = {'name': ni_name, 'type': ni_type}
    return str_path, str_data

def _network_instance_static_route(ni_name, prefix, next_hop, next_hop_index=0) -> Tuple[str, Dict]:
    str_path = '/network_instance[{:s}]/static_route[{:s}]'.format(ni_name, prefix)
    str_data = {'name': ni_name, 'prefix': prefix, 'next_hop': next_hop, 'next_hop_index': next_hop_index}
    return str_path, str_data

def _network_instance_interface(ni_name, if_name, sif_index) -> Tuple[str, Dict]:
    str_path = '/network_instance[{:s}]/interface[{:s}.{:d}]'.format(ni_name, if_name, sif_index)
    str_data = {'name': ni_name, 'if_name': if_name, 'sif_index': sif_index}
    return str_path, str_data

class EndpointComposer:
    def __init__(self, endpoint_uuid : str) -> None:
        self.uuid = endpoint_uuid
        self.objekt : Optional[EndPoint] = None
        self.sub_interface_index = 0
        self.ipv4_address = None
        self.ipv4_prefix = None

    def configure(self, endpoint_obj : EndPoint, settings : Optional[TreeNode]) -> None:
        self.objekt = endpoint_obj
        if settings is None: return
        json_settings : Dict = settings.value
        self.ipv4_address = json_settings['ipv4_address']
        self.ipv4_prefix = json_settings['ipv4_prefix']
        self.sub_interface_index = json_settings['sub_interface_index']

    def get_config_rules(self, network_instance_name : str, delete : bool = False) -> List[Dict]:
        json_config_rule = json_config_rule_delete if delete else json_config_rule_set
        return [
            json_config_rule(*_interface(
                self.objekt.name, self.sub_interface_index, self.ipv4_address, self.ipv4_prefix, True
            )),
            json_config_rule(*_network_instance_interface(
                network_instance_name, self.objekt.name, self.sub_interface_index
            )),
        ]

class DeviceComposer:
    def __init__(self, device_uuid : str) -> None:
        self.uuid = device_uuid
        self.objekt : Optional[Device] = None
        self.endpoints : Dict[str, EndpointComposer] = dict()
        self.static_routes : Dict[str, str] = dict()
    
    def get_endpoint(self, endpoint_uuid : str) -> EndpointComposer:
        if endpoint_uuid not in self.endpoints:
            self.endpoints[endpoint_uuid] = EndpointComposer(endpoint_uuid)
        return self.endpoints[endpoint_uuid]

    def configure(self, device_obj : Device, settings : Optional[TreeNode]) -> None:
        self.objekt = device_obj
        if settings is None: return
        json_settings : Dict = settings.value
        static_routes = json_settings.get('static_routes', [])
        for static_route in static_routes:
            prefix   = static_route['prefix']
            next_hop = static_route['next_hop']
            self.static_routes[prefix] = next_hop

    def get_config_rules(self, network_instance_name : str, delete : bool = False) -> List[Dict]:
        json_config_rule = json_config_rule_delete if delete else json_config_rule_set
        config_rules = [
            json_config_rule(*_network_instance(network_instance_name, 'L3VRF'))
        ]
        for endpoint in self.endpoints.values():
            config_rules.extend(endpoint.get_config_rules(network_instance_name, delete=delete))
        for prefix, next_hop in self.static_routes.items():
            config_rules.append(
                json_config_rule(*_network_instance_static_route(network_instance_name, prefix, next_hop))
            )
        if delete: config_rules = list(reversed(config_rules))
        return config_rules

class ConfigRuleComposer:
    def __init__(self) -> None:
        self.devices : Dict[str, DeviceComposer] = dict()

    def get_device(self, device_uuid : str) -> DeviceComposer:
        if device_uuid not in self.devices:
            self.devices[device_uuid] = DeviceComposer(device_uuid)
        return self.devices[device_uuid]

    def get_config_rules(self, network_instance_name : str, delete : bool = False) -> Dict[str, List[Dict]]:
        return {
            device_uuid : device.get_config_rules(network_instance_name, delete=delete)
            for device_uuid, device in self.devices.items()
        }
