# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from common.tools.object_factory.Context import json_context, json_context_id
from common.tools.object_factory.Topology import json_topology, json_topology_id
from common.proto.context_pb2 import Context, ContextId, Empty, Topology, TopologyId
from context.client.ContextClient import ContextClient

def create_context(
    context_client : ContextClient, context_uuid : str
) -> None:
    existing_context_ids = context_client.ListContextIds(Empty())
    existing_context_uuids = {context_id.context_uuid.uuid for context_id in existing_context_ids.context_ids}
    if context_uuid in existing_context_uuids: return
    context_client.SetContext(Context(**json_context(context_uuid)))

def create_topology(
    context_client : ContextClient, context_uuid : str, topology_uuid : str
) -> None:
    context_id = ContextId(**json_context_id(context_uuid))
    existing_topology_ids = context_client.ListTopologyIds(context_id)
    existing_topology_uuids = {topology_id.topology_uuid.uuid for topology_id in existing_topology_ids.topology_ids}
    if topology_uuid in existing_topology_uuids: return
    context_client.SetTopology(Topology(**json_topology(topology_uuid, context_id=context_id)))

def add_topology_to_context(
    context_client : ContextClient, context_uuid : str, topology_uuid : str
) -> None:
    context_id = ContextId(**json_context_id(context_uuid))
    context_ro = context_client.GetContext(context_id)
    topology_uuids = {topology_id.topology_uuid.uuid for topology_id in context_ro.topology_ids}
    if topology_uuid in topology_uuids: return

    context_rw = Context()
    context_rw.CopyFrom(context_ro)
    context_rw.topology_ids.add().topology_uuid.uuid = topology_uuid # pylint: disable=no-member
    context_client.SetContext(context_rw)

def add_device_to_topology(
    context_client : ContextClient, context_uuid : str, topology_uuid : str, device_uuid : str
) -> None:
    context_id = ContextId(**json_context_id(context_uuid))
    topology_id = TopologyId(**json_topology_id(topology_uuid, context_id=context_id))
    topology_ro = context_client.GetTopology(topology_id)
    device_uuids = {device_id.device_uuid.uuid for device_id in topology_ro.device_ids}
    if device_uuid in device_uuids: return

    topology_rw = Topology()
    topology_rw.CopyFrom(topology_ro)
    topology_rw.device_ids.add().device_uuid.uuid = device_uuid # pylint: disable=no-member
    context_client.SetTopology(topology_rw)

def add_link_to_topology(
    context_client : ContextClient, context_uuid : str, topology_uuid : str, link_uuid : str
) -> None:
    context_id = ContextId(**json_context_id(context_uuid))
    topology_id = TopologyId(**json_topology_id(topology_uuid, context_id=context_id))
    topology_ro = context_client.GetTopology(topology_id)
    link_uuids = {link_id.link_uuid.uuid for link_id in topology_ro.link_ids}
    if link_uuid in link_uuids: return

    topology_rw = Topology()
    topology_rw.CopyFrom(topology_ro)
    topology_rw.link_ids.add().link_uuid.uuid = link_uuid # pylint: disable=no-member
    context_client.SetTopology(topology_rw)
