# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import functools, logging, operator
from enum import Enum
from typing import Dict, List
from common.orm.fields.EnumeratedField import EnumeratedField
from common.orm.fields.ForeignKeyField import ForeignKeyField
from common.orm.fields.PrimaryKeyField import PrimaryKeyField
from common.orm.fields.StringField import StringField
from common.orm.model.Model import Model
from common.orm.HighLevel import get_related_objects
from common.proto.context_pb2 import SliceStatusEnum
from .ConfigRuleModel import ConfigModel
from .ConstraintModel import ConstraintsModel
from .models.ContextModel import ContextModel
from .Tools import grpc_to_enum

LOGGER = logging.getLogger(__name__)

class ORM_SliceStatusEnum(Enum):
    UNDEFINED = SliceStatusEnum.SLICESTATUS_UNDEFINED
    PLANNED   = SliceStatusEnum.SLICESTATUS_PLANNED
    INIT      = SliceStatusEnum.SLICESTATUS_INIT
    ACTIVE    = SliceStatusEnum.SLICESTATUS_ACTIVE
    DEINIT    = SliceStatusEnum.SLICESTATUS_DEINIT

grpc_to_enum__slice_status = functools.partial(
    grpc_to_enum, SliceStatusEnum, ORM_SliceStatusEnum)

class SliceModel(Model):
    pk = PrimaryKeyField()
    context_fk = ForeignKeyField(ContextModel)
    slice_uuid = StringField(required=True, allow_empty=False)
    slice_constraints_fk = ForeignKeyField(ConstraintsModel)
    slice_status = EnumeratedField(ORM_SliceStatusEnum, required=True)
    slice_config_fk = ForeignKeyField(ConfigModel)
    slice_owner_uuid = StringField(required=False, allow_empty=True)
    slice_owner_string = StringField(required=False, allow_empty=True)

    def delete(self) -> None:
        # pylint: disable=import-outside-toplevel
        from .RelationModels import SliceEndPointModel, SliceServiceModel, SliceSubSliceModel

        for db_slice_endpoint_pk,_ in self.references(SliceEndPointModel):
            SliceEndPointModel(self.database, db_slice_endpoint_pk).delete()

        for db_slice_service_pk,_ in self.references(SliceServiceModel):
            SliceServiceModel(self.database, db_slice_service_pk).delete()

        for db_slice_subslice_pk,_ in self.references(SliceSubSliceModel):
            SliceSubSliceModel(self.database, db_slice_subslice_pk).delete()

        super().delete()

        ConfigModel(self.database, self.slice_config_fk).delete()
        ConstraintsModel(self.database, self.slice_constraints_fk).delete()

    def dump_id(self) -> Dict:
        context_id = ContextModel(self.database, self.context_fk).dump_id()
        return {
            'context_id': context_id,
            'slice_uuid': {'uuid': self.slice_uuid},
        }

    def dump_endpoint_ids(self) -> List[Dict]:
        from .RelationModels import SliceEndPointModel # pylint: disable=import-outside-toplevel
        db_endpoints = get_related_objects(self, SliceEndPointModel, 'endpoint_fk')
        return [db_endpoint.dump_id() for db_endpoint in sorted(db_endpoints, key=operator.attrgetter('pk'))]

    def dump_constraints(self) -> List[Dict]:
        return ConstraintsModel(self.database, self.slice_constraints_fk).dump()

    def dump_config(self) -> Dict:
        return ConfigModel(self.database, self.slice_config_fk).dump()

    def dump_service_ids(self) -> List[Dict]:
        from .RelationModels import SliceServiceModel # pylint: disable=import-outside-toplevel
        db_services = get_related_objects(self, SliceServiceModel, 'service_fk')
        return [db_service.dump_id() for db_service in sorted(db_services, key=operator.attrgetter('pk'))]

    def dump_subslice_ids(self) -> List[Dict]:
        from .RelationModels import SliceSubSliceModel # pylint: disable=import-outside-toplevel
        db_subslices = get_related_objects(self, SliceSubSliceModel, 'sub_slice_fk')
        return [
            db_subslice.dump_id()
            for db_subslice in sorted(db_subslices, key=operator.attrgetter('pk'))
            if db_subslice.pk != self.pk # if I'm subslice of other slice, I will appear as subslice of myself
        ]

    def dump(   # pylint: disable=arguments-differ
            self, include_endpoint_ids=True, include_constraints=True, include_config_rules=True,
            include_service_ids=True, include_subslice_ids=True
        ) -> Dict:
        result = {
            'slice_id': self.dump_id(),
            'slice_status': {'slice_status': self.slice_status.value},
        }
        if include_endpoint_ids: result['slice_endpoint_ids'] = self.dump_endpoint_ids()
        if include_constraints: result['slice_constraints'] = self.dump_constraints()
        if include_config_rules: result.setdefault('slice_config', {})['config_rules'] = self.dump_config()
        if include_service_ids: result['slice_service_ids'] = self.dump_service_ids()
        if include_subslice_ids: result['slice_subslice_ids'] = self.dump_subslice_ids()

        if len(self.slice_owner_uuid) > 0:
            result.setdefault('slice_owner', {}).setdefault('owner_uuid', {})['uuid'] = self.slice_owner_uuid

        if len(self.slice_owner_string) > 0:
            result.setdefault('slice_owner', {})['owner_string'] = self.slice_owner_string

        return result
