/*
* Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

package eu.teraflow.policy.model;

import static com.google.common.base.Preconditions.checkArgument;

import eu.teraflow.policy.common.Util;
import java.util.ArrayList;
import java.util.List;

public class PolicyRuleBasic {

    private final String policyRuleId;
    private PolicyRuleState policyRuleState;
    private final int priority;
    private final List<PolicyRuleCondition> policyRuleConditions;
    private final BooleanOperator booleanOperator;
    private final List<PolicyRuleAction> policyRuleActions;
    private final Boolean isValid;
    private final String exceptionMessage;

    public PolicyRuleBasic(
            String policyRuleId,
            PolicyRuleState policyRuleState,
            int priority,
            List<PolicyRuleCondition> policyRuleConditions,
            BooleanOperator booleanOperator,
            List<PolicyRuleAction> policyRuleActions) {

        String policyRuleId_val;
        int priority_val;
        List<PolicyRuleCondition> policyRuleConditions_val;
        BooleanOperator booleanOperator_val;
        List<PolicyRuleAction> policyRuleActions_val;
        Boolean isValid_val;
        String exceptionMessage_val = "";

        try {
            // checkNotNull(policyRuleId, "Policy rule ID must not be null.");
            checkArgument(!policyRuleId.isBlank(), "Policy rule ID must not be empty.");
            policyRuleId_val = policyRuleId;
            this.policyRuleState = policyRuleState;
            checkArgument(priority >= 0, "Priority value must be greater or equal than zero.");
            priority_val = priority;
            // checkNotNull(policyRuleConditions, "Policy Rule conditions cannot be null.");
            checkArgument(!policyRuleConditions.isEmpty(), "Policy Rule conditions cannot be empty.");
            policyRuleConditions_val = policyRuleConditions;
            checkArgument(
                    booleanOperator != BooleanOperator.POLICYRULE_CONDITION_BOOLEAN_UNDEFINED,
                    "Boolean operator cannot be undefined");
            booleanOperator_val = booleanOperator;
            // checkNotNull(policyRuleActions, "Policy Rule actions cannot be null.");
            checkArgument(!policyRuleActions.isEmpty(), "Policy Rule actions cannot be empty.");
            policyRuleActions_val = policyRuleActions;
            isValid_val = true;

        } catch (Exception e) {
            policyRuleId_val = "";
            priority_val = 0;
            policyRuleConditions_val = new ArrayList<PolicyRuleCondition>();
            booleanOperator_val = BooleanOperator.POLICYRULE_CONDITION_BOOLEAN_UNDEFINED;
            policyRuleActions_val = new ArrayList<PolicyRuleAction>();
            isValid_val = false;
            exceptionMessage_val = e.toString();
        }

        this.isValid = isValid_val;
        this.policyRuleId = policyRuleId_val;
        this.priority = priority_val;
        this.policyRuleConditions = policyRuleConditions_val;
        this.booleanOperator = booleanOperator_val;
        this.policyRuleActions = policyRuleActions_val;
        this.exceptionMessage = exceptionMessage_val;
    }

    public boolean areArgumentsValid() {
        return isValid;
    }

    public String getExeceptionMessage() {
        return exceptionMessage;
    }

    public String getPolicyRuleId() {
        return policyRuleId;
    }

    public PolicyRuleState getPolicyRuleState() {
        return policyRuleState;
    }

    public void setPolicyRuleState(PolicyRuleState state) {
        this.policyRuleState = state;
    }

    public int getPriority() {
        return priority;
    }

    public List<PolicyRuleCondition> getPolicyRuleConditions() {
        return policyRuleConditions;
    }

    public BooleanOperator getBooleanOperator() {
        return booleanOperator;
    }

    public List<PolicyRuleAction> getPolicyRuleActions() {
        return policyRuleActions;
    }

    @Override
    public String toString() {
        return String.format(
                "%s:{policyRuleId:\"%s\", %s, priority:%d, [%s], booleanOperator:\"%s\", [%s]}",
                getClass().getSimpleName(),
                policyRuleId,
                policyRuleState,
                priority,
                Util.toString(policyRuleConditions),
                booleanOperator.toString(),
                Util.toString(policyRuleActions));
    }
}
