/*
* Copyright 2022-2023 ETSI TeraFlowSDN - TFS OSG (https://tfs.etsi.org/)
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

package eu.teraflow.automation;

import automation.Automation;
import automation.Automation.DeviceRoleConfig;
import automation.Automation.DeviceRoleState;
import context.ContextOuterClass;
import eu.teraflow.automation.context.model.Device;
import eu.teraflow.automation.model.DeviceRoleId;
import eu.teraflow.automation.model.DeviceState;
import io.quarkus.grpc.GrpcService;
import io.smallrye.mutiny.Uni;
import javax.inject.Inject;

import org.eclipse.microprofile.metrics.MetricUnits;
import org.eclipse.microprofile.metrics.annotation.Counted;
import org.eclipse.microprofile.metrics.annotation.Timed;

@GrpcService
public class AutomationGatewayImpl implements AutomationGateway {

    private final AutomationService automationService;
    private final Serializer serializer;

    @Inject
    public AutomationGatewayImpl(AutomationService automationService, Serializer serializer) {
        this.automationService = automationService;
        this.serializer = serializer;
    }

    @Override
    @Counted(name = "automation_ztpGetDeviceRole_counter")
    @Timed(name = "automation_ztpGetDeviceRole_histogram", unit = MetricUnits.MILLISECONDS)
    public Uni<Automation.DeviceRole> ztpGetDeviceRole(Automation.DeviceRoleId request) {
        return Uni.createFrom()
                .item(() -> Automation.DeviceRole.newBuilder().setDevRoleId(request).build());
    }

    @Override
    @Counted(name = "automation_ztpGetDeviceRolesByDeviceId_counter")
    @Timed(name = "automation_ztpGetDeviceRolesByDeviceId_histogram", unit = MetricUnits.MILLISECONDS)
    public Uni<Automation.DeviceRoleList> ztpGetDeviceRolesByDeviceId(
            ContextOuterClass.DeviceId request) {
        return Uni.createFrom().item(() -> Automation.DeviceRoleList.newBuilder().build());
    }

    @Override
    @Counted(name = "automation_ztpAdd_counter")
    @Timed(name = "automation_ztpAdd_histogram", unit = MetricUnits.MILLISECONDS)
    public Uni<Automation.DeviceRoleState> ztpAdd(Automation.DeviceRole request) {
        final var devRoleId = request.getDevRoleId().getDevRoleId().getUuid();
        final var deviceId = serializer.deserialize(request.getDevRoleId().getDevId());

        return automationService
                .addDevice(deviceId)
                .onItem()
                .transform(device -> transformToDeviceRoleState(device, devRoleId, DeviceState.CREATED));
    }

    @Override
    @Counted(name = "automation_ztpUpdate_counter")
    @Timed(name = "automation_ztpUpdate_histogram", unit = MetricUnits.MILLISECONDS)
    public Uni<DeviceRoleState> ztpUpdate(DeviceRoleConfig request) {
        final var devRoleId = request.getDevRole().getDevRoleId().getDevRoleId().getUuid();
        final var deviceId = serializer.deserialize(request.getDevRole().getDevRoleId().getDevId());
        final var deviceConfig = serializer.deserialize(request.getDevConfig());

        return automationService
                .updateDevice(deviceId, deviceConfig)
                .onItem()
                .transform(device -> transformToDeviceRoleState(device, devRoleId, DeviceState.UPDATED));
    }

    @Override
    @Counted(name = "automation_ztpDelete_counter")
    @Timed(name = "automation_ztpDelete_histogram", unit = MetricUnits.MILLISECONDS)
    public Uni<Automation.DeviceRoleState> ztpDelete(Automation.DeviceRole request) {
        final var devRoleId = request.getDevRoleId().getDevRoleId().getUuid();
        return automationService
                .deleteDevice(devRoleId)
                .onItem()
                .transform(device -> transformToDeviceRoleState(device, devRoleId, DeviceState.DELETED));
    }

    @Override
    @Counted(name = "automation_ztpDeleteAll_counter")
    @Timed(name = "automation_ztpDeleteAll_histogram", unit = MetricUnits.MILLISECONDS)
    public Uni<Automation.DeviceDeletionResult> ztpDeleteAll(ContextOuterClass.Empty empty) {
        return Uni.createFrom().item(() -> Automation.DeviceDeletionResult.newBuilder().build());
    }

    private Automation.DeviceRoleState transformToDeviceRoleState(
            Device device, String devRoleId, DeviceState deviceState) {
        final var deviceRoleId = new DeviceRoleId(devRoleId, device.getDeviceId());
        final var serializeDeviceRoleId = serializer.serialize(deviceRoleId);
        final var serializedDeviceState = serializer.serialize(deviceState);

        return Automation.DeviceRoleState.newBuilder()
                .setDevRoleId(serializeDeviceRoleId)
                .setDevRoleState(serializedDeviceState)
                .build();
    }
}
