# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import json, logging, requests
from device.service.driver_api._Driver import RESOURCE_ENDPOINTS

LOGGER = logging.getLogger(__name__)


def find_key(resource, key):
    return json.loads(resource[1])[key]

# this function exports only the endpoints which are not already involved in a microwave physical link
def isAnExportableEndpoint(node, termination_point_id, links):
    # for each link we check if the endpoint (termination_point_id) is already used by an existing link
    for link in links:
        src = link['source']
        dest = link['destination']
        if dest['dest-node'] == node and dest['dest-tp'] == termination_point_id:
            return False
        elif src['source-node'] == node and src['source-tp'] == termination_point_id:
            return False

    return True

def config_getter(root_url, resource_key, timeout):
    # getting endpoints
    url = '{:s}/nmswebs/restconf/data/ietf-network:networks/network=SIAE-ETH-TOPOLOGY?fields=ietf-network-topology:link(link-id;destination(dest-node;dest-tp);source(source-node;source-tp));node(node-id;ietf-network-topology:termination-point(tp-id;ietf-te-topology:te/name)))'.format(root_url)
    result = []
    try:
        response = requests.get(url, timeout=timeout, verify=False)
    except requests.exceptions.Timeout:
        LOGGER.exception('Timeout connecting {:s}'.format(url))
    except Exception as e:  # pylint: disable=broad-except
        LOGGER.exception('Exception retrieving {:s}'.format(resource_key))
        result.append((resource_key, e))
    else:
        context = json.loads(response.content)
    

        if resource_key == RESOURCE_ENDPOINTS:
            if not (context.get('ietf-network:network') is None):
                network_instance = context['ietf-network:network']
                if not (network_instance.get('ietf-network-topology:link') is None):
                    links = network_instance['ietf-network-topology:link']  

                    for sip in network_instance['node']:
                        tp = sip['ietf-network-topology:termination-point']
                        node_id = sip['node-id']
                        for te in tp:
                            tp_id = te['tp-id']
                            if isAnExportableEndpoint(node_id, tp_id, links):
                                result.append(('/endpoints/endpoint[{:s}:{:s}]'.format(node_id,tp_id), {'uuid': tp_id, 'type': te['ietf-te-topology:te']['name']}))

    # getting created services 
    url = '{:s}/nmswebs/restconf/data/ietf-eth-tran-service:etht-svc'.format(root_url)
    try:
        response = requests.get(url, timeout=timeout, verify=False)
    except requests.exceptions.Timeout:
        LOGGER.exception('Timeout connecting {:s}'.format(url))
    except Exception as e:  # pylint: disable=broad-except
        LOGGER.exception('Exception retrieving {:s}'.format(resource_key))
        result.append((resource_key, e))
    else:
        context = json.loads(response.content)
    

        if resource_key == RESOURCE_ENDPOINTS:
            if not (context.get('ietf-eth-tran-service:etht-svc') is None):
                etht_service = context['ietf-eth-tran-service:etht-svc']
                if not (etht_service.get('etht-svc-instances') is None):
                    service_instances = etht_service['etht-svc-instances'] 
            
                    for service in service_instances:
                        service_name = service['etht-svc-name']
                        result.append(('/services/service[{:s}]'.format(service_name), {'uuid': service_name, 'type': service['etht-svc-type']}))
            
    return result


def create_connectivity_service(
    root_url, timeout, uuid, node_id_src, tp_id_src, node_id_dst, tp_id_dst, vlan_id):

    url = '{:s}/nmswebs/restconf/data/ietf-eth-tran-service:etht-svc'.format(root_url)
    headers = {'content-type': 'application/json'}
    data = {}
    
    data = {
            "etht-svc-instances": 
            [
	            {
		            "etht-svc-name": uuid,
		            "etht-svc-type": "ietf-eth-tran-types:p2p-svc",
		            "etht-svc-end-points": 
                    [
		                {
			                "etht-svc-access-points": 
                            [
			                    {       
			                        "access-node-id": node_id_src,
			                        "access-ltp-id": tp_id_src,
			                        "access-point-id": "1"
			                    }
			                ],
		                    "outer-tag": 
                                {
			                        "vlan-value": vlan_id,
			                        "tag-type": "ietf-eth-tran-types:classify-c-vlan"
		                        },
		                    "etht-svc-end-point-name": node_id_src+':'+str(tp_id_src),
		                    "service-classification-type": "ietf-eth-tran-types:vlan-classification"
		                },
		                {
		                    "etht-svc-access-points": 
                            [
		                        {
			                        "access-node-id": node_id_dst,
			                        "access-ltp-id": tp_id_dst,
			                        "access-point-id": "2"
		                        }
		                    ],
		                    "outer-tag": 
                                {
			                        "vlan-value": vlan_id,
			                        "tag-type": "ietf-eth-tran-types:classify-c-vlan"
		                        },
		                    "etht-svc-end-point-name": node_id_dst+':'+str(tp_id_dst),
		                    "service-classification-type": "ietf-eth-tran-types:vlan-classification"
		                }
		            ]	
	            }
	        ]       
           }

    
    results = []
    try:
        LOGGER.info('Connectivity service {:s}: {:s}'.format(str(uuid), str(data)))
        response = requests.post(url=url, data=json.dumps(data), timeout=timeout, headers=headers, verify=False)
        LOGGER.info('Microwave Driver response: {:s}'.format(str(response)))
    except Exception as e:  # pylint: disable=broad-except
        LOGGER.exception('Exception creating ConnectivityService(uuid={:s}, data={:s})'.format(str(uuid), str(data)))
        results.append(e)
    else:
        if response.status_code != 201:
            msg = 'Could not create ConnectivityService(uuid={:s}, data={:s}). status_code={:s} reply={:s}'
            LOGGER.error(msg.format(str(uuid), str(data), str(response.status_code), str(response)))
        results.append(response.status_code == 201)
    return results


def delete_connectivity_service(root_url, timeout, uuid):
    url = '{:s}/nmswebs/restconf/data/ietf-eth-tran-service:etht-svc/etht-svc-instances={:s}'
    url = url.format(root_url, uuid)
    results = []
    try:
        response = requests.delete(url=url, timeout=timeout, verify=False)
    except Exception as e:  # pylint: disable=broad-except
        LOGGER.exception('Exception deleting ConnectivityService(uuid={:s})'.format(str(uuid)))
        results.append(e)
    else:
        if response.status_code != 201:
            msg = 'Could not delete ConnectivityService(uuid={:s}). status_code={:s} reply={:s}'
            LOGGER.error(msg.format(str(uuid), str(response.status_code), str(response)))
        results.append(response.status_code == 202)
    return results
