from __future__ import annotations
from typing import TYPE_CHECKING, Dict
from ..entity._Entity import _Entity
from ..entity.EntityAttributes import EntityAttributes
from .Keys import KEY_ENDPOINT

if TYPE_CHECKING:
    from .Context import Context
    from .Topology import Topology
    from .Device import Device

VALIDATORS = {
    'port_type': lambda v: v is None or isinstance(v, str),
}

class Endpoint(_Entity):
    def __init__(self, endpoint_uuid : str, parent : 'Device'):
        super().__init__(endpoint_uuid, parent=parent)
        self._attributes = EntityAttributes(self, KEY_ENDPOINT, VALIDATORS)

    @property
    def parent(self) -> 'Device': return self._parent

    @property
    def context(self) -> 'Context': return self.parent.context

    @property
    def context_uuid(self) -> str: return self.parent.context_uuid

    @property
    def topology(self) -> 'Topology': return self.parent.topology

    @property
    def topology_uuid(self) -> str: return self.parent.topology_uuid

    @property
    def device(self) -> 'Device': return self.parent

    @property
    def device_uuid(self) -> str: return self.parent.device_uuid

    @property
    def endpoint_uuid(self) -> str: return self._entity_uuid

    @property
    def attributes(self) -> EntityAttributes: return self._attributes

    def create(self, port_type : str) -> 'Endpoint':
        self.update(update_attributes={
            'port_type': port_type
        })
        self.parent.endpoints.add(self.endpoint_uuid)
        return self

    def update(self, update_attributes={}, remove_attributes=[]) -> 'Endpoint':
        self.attributes.update(update_attributes=update_attributes, remove_attributes=remove_attributes)
        return self

    def delete(self) -> None:
        self.attributes.delete()
        self.parent.endpoints.delete(self.endpoint_uuid)

    def dump_id(self) -> Dict:
        return {
            'topoId': self.topology.dump_id(),
            'dev_id': self.device.dump_id(),
            'port_id': {'uuid': self.endpoint_uuid},
        }

    def dump(self) -> Dict:
        attributes = self.attributes.get()
        return {
            'port_id': self.dump_id(),
            'port_type': attributes.get('port_type', None),
        }
