from typing import Dict, List, Set, Tuple
import grpc, logging
from prometheus_client import Counter, Histogram
from common.Checkers import chk_options, chk_string
from common.database.api.Database import Database
from common.database.api.context.Constants import DEFAULT_CONTEXT_ID, DEFAULT_TOPOLOGY_ID
from common.database.api.context.service.ServiceState import ServiceState, servicestate_enum_values, \
    to_servicestate_enum
from common.database.api.context.service.ServiceType import ServiceType, servicetype_enum_values, to_servicetype_enum
from common.exceptions.ServiceException import ServiceException
from service.proto.context_pb2 import Empty
from service.proto.service_pb2 import ConnectionList, Service, ServiceId, ServiceList
from service.proto.service_pb2_grpc import ServiceServiceServicer

LOGGER = logging.getLogger(__name__)

GETSERVICELIST_COUNTER_STARTED    = Counter  ('service_getservicelist_counter_started',
                                              'Service:GetServiceList counter of requests started'  )
GETSERVICELIST_COUNTER_COMPLETED  = Counter  ('service_getservicelist_counter_completed',
                                              'Service:GetServiceList counter of requests completed')
GETSERVICELIST_COUNTER_FAILED     = Counter  ('service_getservicelist_counter_failed',
                                              'Service:GetServiceList counter of requests failed'   )
GETSERVICELIST_HISTOGRAM_DURATION = Histogram('service_getservicelist_histogram_duration',
                                              'Service:GetServiceList histogram of request duration')

CREATESERVICE_COUNTER_STARTED    = Counter  ('service_createservice_counter_started',
                                             'Service:CreateService counter of requests started'  )
CREATESERVICE_COUNTER_COMPLETED  = Counter  ('service_createservice_counter_completed',
                                             'Service:CreateService counter of requests completed')
CREATESERVICE_COUNTER_FAILED     = Counter  ('service_createservice_counter_failed',
                                             'Service:CreateService counter of requests failed'   )
CREATESERVICE_HISTOGRAM_DURATION = Histogram('service_createservice_histogram_duration',
                                             'Service:CreateService histogram of request duration')

UPDATESERVICE_COUNTER_STARTED    = Counter  ('service_updateservice_counter_started',
                                             'Service:UpdateService counter of requests started'  )
UPDATESERVICE_COUNTER_COMPLETED  = Counter  ('service_updateservice_counter_completed',
                                             'Service:UpdateService counter of requests completed')
UPDATESERVICE_COUNTER_FAILED     = Counter  ('service_updateservice_counter_failed',
                                             'Service:UpdateService counter of requests failed'   )
UPDATESERVICE_HISTOGRAM_DURATION = Histogram('service_updateservice_histogram_duration',
                                             'Service:UpdateService histogram of request duration')

DELETESERVICE_COUNTER_STARTED    = Counter  ('service_deleteservice_counter_started',
                                             'Service:DeleteService counter of requests started'  )
DELETESERVICE_COUNTER_COMPLETED  = Counter  ('service_deleteservice_counter_completed',
                                             'Service:DeleteService counter of requests completed')
DELETESERVICE_COUNTER_FAILED     = Counter  ('service_deleteservice_counter_failed',
                                             'Service:DeleteService counter of requests failed'   )
DELETESERVICE_HISTOGRAM_DURATION = Histogram('service_deleteservice_histogram_duration',
                                             'Service:DeleteService histogram of request duration')

GETSERVICEBYID_COUNTER_STARTED    = Counter  ('service_getservicebyid_counter_started',
                                              'Service:GetServiceById counter of requests started'  )
GETSERVICEBYID_COUNTER_COMPLETED  = Counter  ('service_getservicebyid_counter_completed',
                                              'Service:GetServiceById counter of requests completed')
GETSERVICEBYID_COUNTER_FAILED     = Counter  ('service_getservicebyid_counter_failed',
                                              'Service:GetServiceById counter of requests failed'   )
GETSERVICEBYID_HISTOGRAM_DURATION = Histogram('service_getservicebyid_histogram_duration',
                                              'Service:GetServiceById histogram of request duration')

GETCONNECTIONLIST_COUNTER_STARTED    = Counter  ('service_getconnectionlist_counter_started',
                                                 'Service:GetConnectionList counter of requests started'  )
GETCONNECTIONLIST_COUNTER_COMPLETED  = Counter  ('service_getconnectionlist_counter_completed',
                                                 'Service:GetConnectionList counter of requests completed')
GETCONNECTIONLIST_COUNTER_FAILED     = Counter  ('service_getconnectionlist_counter_failed',
                                                 'Service:GetConnectionList counter of requests failed'   )
GETCONNECTIONLIST_HISTOGRAM_DURATION = Histogram('service_getconnectionlist_histogram_duration',
                                                 'Service:GetConnectionList histogram of request duration')

class ServiceServiceServicerImpl(ServiceServiceServicer):
    def __init__(self, database : Database):
        LOGGER.debug('Creating Servicer...')
        self.database = database
        LOGGER.debug('Servicer Created')

    @GETSERVICELIST_HISTOGRAM_DURATION.time()
    def GetServiceList(self, request : Empty, grpc_context : grpc.ServicerContext) -> ServiceList:
        GETSERVICELIST_COUNTER_STARTED.inc()
        try:
            LOGGER.debug('GetServiceList request: {}'.format(str(request)))

            # ----- Validate request data and pre-conditions -----------------------------------------------------------

            # ----- Retrieve data from the database --------------------------------------------------------------------
            db_context_uuids = self.database.contexts
            json_services = []
            for db_context_uuid in db_context_uuids:
                json_services.extend(self.database.context(db_context_uuid).dump_services())

            # ----- Compose reply --------------------------------------------------------------------------------------
            reply = ServiceList(cs=json_services)
            LOGGER.debug('GetServiceList reply: {}'.format(str(reply)))
            GETSERVICELIST_COUNTER_COMPLETED.inc()
            return reply
        except ServiceException as e:
            grpc_context.abort(e.code, e.details)
        except Exception as e:                                      # pragma: no cover
            LOGGER.exception('GetServiceList exception')            # pragma: no cover
            GETSERVICELIST_COUNTER_FAILED.inc()                     # pragma: no cover
            grpc_context.abort(grpc.StatusCode.INTERNAL, str(e))    # pragma: no cover

    @CREATESERVICE_HISTOGRAM_DURATION.time()
    def CreateService(self, request : Service, grpc_context : grpc.ServicerContext) -> ServiceId:
        CREATESERVICE_COUNTER_STARTED.inc()
        try:
            LOGGER.debug('CreateService request: {}'.format(str(request)))

            # ----- Validate request data and pre-conditions -----------------------------------------------------------
            try:
                context_id     = chk_string ('service.cs_id.contextId.contextUuid.uuid',
                                            request.cs_id.contextId.contextUuid.uuid,
                                            allow_empty=False)
                service_id     = chk_string ('service.cs_id.cs_id.uuid',
                                            request.cs_id.cs_id.uuid,
                                            allow_empty=False)
                service_type   = chk_options('service.serviceType',
                                            request.serviceType,
                                            servicetype_enum_values())
                service_config = chk_string ('service.serviceConfig.serviceConfig',
                                            request.serviceConfig.serviceConfig,
                                            allow_empty=True)
                service_state  = chk_options('device.serviceState.serviceState',
                                            request.serviceState.serviceState,
                                            servicestate_enum_values())
            except Exception as e:
                LOGGER.exception('Invalid arguments:')
                raise ServiceException(grpc.StatusCode.INVALID_ARGUMENT, str(e))

            service_type = to_servicetype_enum(service_type)
            # should not happen because gRPC limits accepted values in enums
            if service_type is None:                                            # pragma: no cover
                msg = 'Unsupported ServiceType({}).'                            # pragma: no cover
                msg = msg.format(request.serviceType)                           # pragma: no cover
                raise ServiceException(grpc.StatusCode.INVALID_ARGUMENT, msg)   # pragma: no cover

            if service_type == ServiceType.UNKNOWN:
                msg = ' '.join([
                    'Service has to be created with a known ServiceType.',
                    'UNKNOWN is only for internal use.',
                ])
                raise ServiceException(grpc.StatusCode.INVALID_ARGUMENT, msg)

            service_state = to_servicestate_enum(service_state)
            # should not happen because gRPC limits accepted values in enums
            if service_state is None:                                           # pragma: no cover
                msg = 'Unsupported ServiceState({}).'                           # pragma: no cover
                msg = msg.format(request.serviceState.serviceState)             # pragma: no cover
                raise ServiceException(grpc.StatusCode.INVALID_ARGUMENT, msg)   # pragma: no cover

            if service_state != ServiceState.PLANNED:
                msg = ' '.join([
                    'Service has to be created with PLANNED state.',
                    'State will be updated by the service appropriately.',
                ])
                raise ServiceException(grpc.StatusCode.INVALID_ARGUMENT, msg)

            db_context = self.database.context(context_id).create()
            if db_context.services.contains(service_id):
                msg = 'Context({})/Service({}) already exists in the database.'
                msg = msg.format(context_id, service_id)
                raise ServiceException(grpc.StatusCode.ALREADY_EXISTS, msg)

            added_context_topology_devices_endpoints : Dict[str, Dict[str, Dict[str, Set[str]]]] = {}
            context_topology_device_endpoint_tuples : List[Tuple[str, str, str, str]] = []
            for i,endpoint in enumerate(request.endpointList):
                try:
                    ep_context_id  = chk_string('endpoint[#{}].topoId.contextId.contextUuid.uuid'.format(i),
                                                endpoint.topoId.contextId.contextUuid.uuid,
                                                allow_empty=True)
                    ep_topology_id = chk_string('endpoint[#{}].topoId.topoId.uuid'.format(i),
                                                endpoint.topoId.topoId.uuid,
                                                allow_empty=True)
                    ep_device_id   = chk_string('endpoint[#{}].dev_id.device_id.uuid'.format(i),
                                                endpoint.dev_id.device_id.uuid,
                                                allow_empty=False)
                    ep_port_id     = chk_string('endpoint[#{}].port_id.uuid'.format(i),
                                                endpoint.port_id.uuid,
                                                allow_empty=False)
                except Exception as e:
                    LOGGER.exception('Invalid arguments:')
                    raise ServiceException(grpc.StatusCode.INVALID_ARGUMENT, str(e))

                if len(ep_context_id) == 0: ep_context_id = DEFAULT_CONTEXT_ID
                if len(ep_topology_id) == 0: ep_topology_id = DEFAULT_TOPOLOGY_ID

                added_devices = added_context_topology_devices_endpoints.get(ep_context_id, {}).get(ep_topology_id, {})
                if ep_device_id in added_devices:
                    msg = 'Duplicated Device({}) in Endpoint(#{}) of Service({}).'
                    msg = msg.format(ep_device_id, i, service_id)
                    raise ServiceException(grpc.StatusCode.INVALID_ARGUMENT, msg)

                if not self.database.contexts.contains(ep_context_id):
                    msg = ' '.join([
                        'Context({}) in Endpoint(#{}) of Service({})',
                        'does not exist in the database.',
                    ])
                    msg = msg.format(ep_context_id, i, service_id)
                    raise ServiceException(grpc.StatusCode.NOT_FOUND, msg)

                db_ep_context = self.database.context(ep_context_id)
                if not db_ep_context.topologies.contains(ep_topology_id):
                    msg = ' '.join([
                        'Context({})/Topology({}) in Endpoint(#{}) of Service({})',
                        'does not exist in the database.',
                    ])
                    msg = msg.format(ep_context_id, ep_topology_id, i, service_id)
                    raise ServiceException(grpc.StatusCode.NOT_FOUND, msg)

                db_ep_topology = db_ep_context.topology(ep_topology_id)
                if not db_ep_topology.devices.contains(ep_device_id):
                    msg = ' '.join([
                        'Context({})/Topology({})/Device({}) in Endpoint(#{}) of Service({})',
                        'does not exist in the database.',
                    ])
                    msg = msg.format(ep_context_id, ep_topology_id, ep_device_id, i, service_id)
                    raise ServiceException(grpc.StatusCode.NOT_FOUND, msg)

                added_device_and_endpoints = added_context_topology_devices_endpoints\
                    .setdefault(ep_context_id, {})\
                    .setdefault(ep_topology_id, {})\
                    .setdefault(ep_device_id, {})

                # should never happen since same device cannot appear 2 times in the service
                if ep_port_id in added_device_and_endpoints:                                # pragma: no cover
                    msg = 'Duplicated Context({})/Topology({})/Device({})/Port({}) in Endpoint(#{}) of Service({}).'
                    msg = msg.format(ep_context_id, ep_topology_id, ep_device_id, ep_port_id, i, service_id)
                    raise ServiceException(grpc.StatusCode.NOT_FOUND, msg)

                db_ep_device = db_ep_topology.device(ep_device_id)
                if not db_ep_device.endpoints.contains(ep_port_id):
                    msg = ' '.join([
                        'Context({})/Topology({})/Device({})/Port({}) in Endpoint(#{}) of Service({})',
                        'does not exist in the database.',
                    ])
                    msg = msg.format(ep_context_id, ep_topology_id, ep_device_id, ep_port_id, i, service_id)
                    raise ServiceException(grpc.StatusCode.NOT_FOUND, msg)

                added_device_and_endpoints.add(ep_port_id)
                context_topology_device_endpoint_tuples.append(
                    (ep_context_id, ep_topology_id, ep_device_id, ep_port_id))

            # ----- Implement changes in the database ------------------------------------------------------------------
            db_service = db_context.service(service_id).create(service_type, service_config, service_state)
            for context_id,topology_id,device_id,endpoint_id in context_topology_device_endpoint_tuples:
                service_endpoint_id = '{}:{}/{}'.format(topology_id, device_id, endpoint_id)
                db_endpoint = db_context.topology(topology_id).device(device_id).endpoint(endpoint_id)
                db_service.endpoint(service_endpoint_id).create(db_endpoint)
            for cons_type,cons_value in constraint_pairs: db_service.constraint(cons_type).create(cons_value)

            # ----- Compose reply --------------------------------------------------------------------------------------
            reply = ServiceId(**db_service.dump_id())
            LOGGER.debug('CreateService reply: {}'.format(str(reply)))
            CREATESERVICE_COUNTER_COMPLETED.inc()
            return reply
        except ServiceException as e:
            grpc_context.abort(e.code, e.details)
        except Exception as e:                                      # pragma: no cover
            LOGGER.exception('CreateService exception')             # pragma: no cover
            CREATESERVICE_COUNTER_FAILED.inc()                      # pragma: no cover
            grpc_context.abort(grpc.StatusCode.INTERNAL, str(e))    # pragma: no cover

    @UPDATESERVICE_HISTOGRAM_DURATION.time()
    def UpdateService(self, request : Service, grpc_context : grpc.ServicerContext) -> ServiceId:
        UPDATESERVICE_COUNTER_STARTED.inc()
        try:
            LOGGER.debug('UpdateService request: {}'.format(str(request)))


            reply = None
            LOGGER.debug('UpdateService reply: {}'.format(str(reply)))
            UPDATESERVICE_COUNTER_COMPLETED.inc()
            return reply
        except ServiceException as e:
            grpc_context.abort(e.code, e.details)
        except Exception as e:                                      # pragma: no cover
            LOGGER.exception('UpdateService exception')             # pragma: no cover
            UPDATESERVICE_COUNTER_FAILED.inc()                      # pragma: no cover
            grpc_context.abort(grpc.StatusCode.INTERNAL, str(e))    # pragma: no cover

    @DELETESERVICE_HISTOGRAM_DURATION.time()
    def DeleteService(self, request : Service, grpc_context : grpc.ServicerContext) -> ServiceId:
        DELETESERVICE_COUNTER_STARTED.inc()
        try:
            LOGGER.debug('DeleteService request: {}'.format(str(request)))


            reply = None
            LOGGER.debug('DeleteService reply: {}'.format(str(reply)))
            DELETESERVICE_COUNTER_COMPLETED.inc()
            return reply
        except ServiceException as e:
            grpc_context.abort(e.code, e.details)
        except Exception as e:                                      # pragma: no cover
            LOGGER.exception('DeleteService exception')             # pragma: no cover
            DELETESERVICE_COUNTER_FAILED.inc()                      # pragma: no cover
            grpc_context.abort(grpc.StatusCode.INTERNAL, str(e))    # pragma: no cover

    @GETSERVICEBYID_HISTOGRAM_DURATION.time()
    def GetServiceById(self, request : ServiceId, grpc_context : grpc.ServicerContext) -> Service:
        GETSERVICEBYID_COUNTER_STARTED.inc()
        try:
            LOGGER.debug('GetServiceById request: {}'.format(str(request)))


            reply = None
            LOGGER.debug('GetServiceById reply: {}'.format(str(reply)))
            GETSERVICEBYID_COUNTER_COMPLETED.inc()
            return reply
        except ServiceException as e:
            grpc_context.abort(e.code, e.details)
        except Exception as e:                                      # pragma: no cover
            LOGGER.exception('GetServiceById exception')            # pragma: no cover
            GETSERVICEBYID_COUNTER_FAILED.inc()                     # pragma: no cover
            grpc_context.abort(grpc.StatusCode.INTERNAL, str(e))    # pragma: no cover

    @GETCONNECTIONLIST_HISTOGRAM_DURATION.time()
    def GetConnectionList(self, request : Empty, grpc_context : grpc.ServicerContext) -> ConnectionList:
        GETCONNECTIONLIST_COUNTER_STARTED.inc()
        try:
            LOGGER.debug('GetConnectionList request: {}'.format(str(request)))


            reply = None
            LOGGER.debug('GetConnectionList reply: {}'.format(str(reply)))
            GETCONNECTIONLIST_COUNTER_COMPLETED.inc()
            return reply
        except ServiceException as e:
            grpc_context.abort(e.code, e.details)
        except Exception as e:                                      # pragma: no cover
            LOGGER.exception('GetConnectionList exception')         # pragma: no cover
            GETCONNECTIONLIST_COUNTER_FAILED.inc()                  # pragma: no cover
            grpc_context.abort(grpc.StatusCode.INTERNAL, str(e))    # pragma: no cover
