import logging, os
from enum import Enum
from .api.Database import Database
from .engines.inmemory.InMemoryDatabaseEngine import InMemoryDatabaseEngine
from .engines.redis.RedisDatabaseEngine import RedisDatabaseEngine

LOGGER = logging.getLogger(__name__)

class DatabaseEngineEnum(Enum):
    INMEMORY = 'inmemory'
    REDIS = 'redis'
    #MONGO = 'mongo'
    #RETHINK = 'rethink'
    #ETCD = 'etcd'

ENGINES = {
    DatabaseEngineEnum.INMEMORY.value: InMemoryDatabaseEngine,
    DatabaseEngineEnum.REDIS.value: RedisDatabaseEngine,
    #DatabaseEngineEnum.MONGO.value: MongoDatabase,
    #DatabaseEngineEnum.RETHINK.value: RethinkDatabase,
    #DatabaseEngineEnum.ETCD.value: EtcdDatabase,
}

DEFAULT_DB_ENGINE = DatabaseEngineEnum.INMEMORY

def get_database(**settings) -> Database:
    engine = os.environ.get('DB_ENGINE', DEFAULT_DB_ENGINE)
    engine = settings.pop('DB_ENGINE', engine)
    if engine is None: raise Exception('Database Engine not specified')
    if isinstance(engine, DatabaseEngineEnum): engine = engine.value
    engine_class = ENGINES.get(engine)
    if engine_class is None: raise Exception('Unsupported DatabaseEngine({})'.format(engine))
    LOGGER.info('Selected Database Engine: {}'.format(engine))
    return Database(engine_class(**settings))
