# Copyright 2022-2023 ETSI TeraFlowSDN - TFS OSG (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

from flask.json import jsonify
from common.proto.context_pb2 import (
    ConnectionId, ContextId, DeviceId, Link, LinkId, ServiceId, SliceId, TopologyId, Service, ServiceStatusEnum
)
from common.proto.policy_pb2 import PolicyRuleId
from common.tools.grpc.Tools import grpc_message_to_json
from common.tools.object_factory.Connection import json_connection_id
from common.tools.object_factory.Context import json_context_id
from common.tools.object_factory.ConfigRule import json_config_rule
from common.tools.object_factory.Constraint import json_constraint_custom
from common.tools.object_factory.EndPoint import json_endpoint_id
from common.tools.object_factory.Device import json_device_id
from common.tools.object_factory.Link import json_link_id, json_link
from common.tools.object_factory.PolicyRule import json_policyrule_id
from common.tools.object_factory.Service import json_service_id, json_service
from common.tools.object_factory.Slice import json_slice_id
from common.tools.object_factory.Topology import json_topology_id


def format_grpc_to_json(grpc_reply):
    return jsonify(grpc_message_to_json(grpc_reply))

def grpc_connection_id(connection_uuid):
    return ConnectionId(**json_connection_id(connection_uuid))

def grpc_context_id(context_uuid):
    return ContextId(**json_context_id(context_uuid))

def grpc_device_id(device_uuid):
    return DeviceId(**json_device_id(device_uuid))

def grpc_link_id(link_uuid):
    return LinkId(**json_link_id(link_uuid))

def grpc_link(link):
    return Link(**json_link(link.link_id.uuid, link.link_endpoint_ids))

def grpc_service_id(context_uuid, service_uuid):
    return ServiceId(**json_service_id(service_uuid, context_id=json_context_id(context_uuid)))

def grpc_service(
    service_uuid, service_type, context_uuid, status=None, endpoint_ids=None, constraints=None, config_rules=None
):
    json_context = json_context_id(context_uuid)
    json_status = status if status else ServiceStatusEnum.SERVICESTATUS_PLANNED
    json_endpoints_ids = [
        json_endpoint_id(
            json_device_id(endpoint_id['device_id']['device_uuid']['uuid']),
            endpoint_id['endpoint_uuid']['uuid']
        )
        for endpoint_id in endpoint_ids
    ] if endpoint_ids else []
    json_constraints = [
        json_constraint_custom(
            constraint['custom']['constraint_type'],
            constraint['custom']['constraint_value']
        )
        for constraint in constraints
    ] if constraints else []
    json_config_rules = [
        json_config_rule(
            config_rule['action'],
            config_rule['custom']['resource_key'],
            config_rule['custom']['resource_value']
        )
        for config_rule in config_rules
    ] if config_rules else []
    return Service(**json_service(
        service_uuid, service_type, json_context, json_status,
        json_endpoints_ids, json_constraints, json_config_rules))

def grpc_slice_id(context_uuid, slice_uuid):
    return SliceId(**json_slice_id(slice_uuid, context_id=json_context_id(context_uuid)))
    
def grpc_topology_id(context_uuid, topology_uuid):
    return TopologyId(**json_topology_id(topology_uuid, context_id=json_context_id(context_uuid)))

def grpc_policy_rule_id(policy_rule_uuid):
    return PolicyRuleId(**json_policyrule_id(policy_rule_uuid))
