# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import json, logging
from typing import List, Optional, Tuple
from common.Constants import DEFAULT_CONTEXT_UUID, DEFAULT_TOPOLOGY_UUID, INTERDOMAIN_TOPOLOGY_UUID
from common.proto.context_pb2 import ContextId, Device, EndPointId, Slice, SliceStatusEnum
from common.tools.context_queries.InterDomain import get_local_domain_devices
from common.tools.grpc.Tools import grpc_message_to_json_string
from common.tools.object_factory.Context import json_context_id
from context.client.ContextClient import ContextClient

LOGGER = logging.getLogger(__name__)

def compute_slice_owner(
    context_client : ContextClient, traversed_domains : List[Tuple[str, Device, bool, List[EndPointId]]]
) -> Optional[str]:
    traversed_domain_uuids = {traversed_domain[0] for traversed_domain in traversed_domains}

    existing_topology_ids = context_client.ListTopologyIds(ContextId(**json_context_id(DEFAULT_CONTEXT_UUID)))
    existing_topology_uuids = {
        topology_id.topology_uuid.uuid for topology_id in existing_topology_ids.topology_ids
    }
    existing_topology_uuids.discard(DEFAULT_TOPOLOGY_UUID)
    existing_topology_uuids.discard(INTERDOMAIN_TOPOLOGY_UUID)

    candidate_owner_uuids = traversed_domain_uuids.intersection(existing_topology_uuids)
    if len(candidate_owner_uuids) != 1:
        data = {
            'traversed_domain_uuids' : [td_uuid for td_uuid in traversed_domain_uuids ],
            'existing_topology_uuids': [et_uuid for et_uuid in existing_topology_uuids],
            'candidate_owner_uuids'  : [co_uuid for co_uuid in candidate_owner_uuids  ],
        }
        LOGGER.warning('Unable to identify slice owner: {:s}'.format(json.dumps(data)))
        return None

    return candidate_owner_uuids.pop()

def compose_slice(
    context_uuid : str, slice_uuid : str, endpoint_ids : List[EndPointId], owner_uuid : Optional[str] = None
) -> Slice:
    slice_ = Slice()
    slice_.slice_id.context_id.context_uuid.uuid = context_uuid            # pylint: disable=no-member
    slice_.slice_id.slice_uuid.uuid = slice_uuid                           # pylint: disable=no-member
    slice_.slice_status.slice_status = SliceStatusEnum.SLICESTATUS_PLANNED # pylint: disable=no-member

    if owner_uuid is not None:
        slice_.slice_owner.owner_uuid.uuid = owner_uuid                    # pylint: disable=no-member

    for endpoint_id in endpoint_ids:
        slice_.slice_endpoint_ids.append(endpoint_id)                      # pylint: disable=no-member

    return slice_
