#!/bin/bash
# Copyright 2022-2023 ETSI TeraFlowSDN - TFS OSG (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


########################################################################################################################
# Read deployment settings
########################################################################################################################

# If not already set, set the namespace where QuestDB will be deployed.
export QDB_NAMESPACE=${QDB_NAMESPACE:-"qdb"}

# If not already set, set the database username to be used by Monitoring.
export QDB_USERNAME=${QDB_USERNAME:-"admin"}

# If not already set, set the database user's password to be used by Monitoring.
export QDB_PASSWORD=${QDB_PASSWORD:-"quest"}

# If not already set, set the table name to be used by Monitoring.
export QDB_TABLE=${QDB_TABLE:-"tfs_monitoring"}

## If not already set, disable flag for dropping table if exists.
## WARNING: ACTIVATING THIS FLAG IMPLIES LOOSING THE TABLE INFORMATION!
## If QDB_DROP_TABLE_IF_EXISTS is "YES", the table pointed by variable QDB_TABLE will be dropped while
## checking/deploying QuestDB.
#export QDB_DROP_TABLE_IF_EXISTS=${QDB_DROP_TABLE_IF_EXISTS:-""}

# If not already set, disable flag for re-deploying QuestDB from scratch.
# WARNING: ACTIVATING THIS FLAG IMPLIES LOOSING THE DATABASE INFORMATION!
# If QDB_REDEPLOY is "YES", the database will be dropped while checking/deploying QuestDB.
export QDB_REDEPLOY=${QDB_REDEPLOY:-""}


########################################################################################################################
# Automated steps start here
########################################################################################################################

# Constants
TMP_FOLDER="./tmp"
QDB_MANIFESTS_PATH="manifests/questdb"

# Create a tmp folder for files modified during the deployment
TMP_MANIFESTS_FOLDER="$TMP_FOLDER/manifests"
TMP_LOGS_FOLDER="$TMP_FOLDER/logs"
QDB_LOG_FILE="$TMP_LOGS_FOLDER/qdb_deploy.log"
mkdir -p $TMP_LOGS_FOLDER

function qdb_deploy() {
    echo "QuestDB Namespace"
    echo ">>> Create QuestDB Namespace (if missing)"
    kubectl create namespace ${QDB_NAMESPACE}
    echo

    echo "QuestDB"
    echo ">>> Checking if QuestDB is deployed..."
    if kubectl get --namespace ${QDB_NAMESPACE} statefulset/questdb &> /dev/null; then
        echo ">>> QuestDB is present; skipping step."
    else
        echo ">>> Deploy QuestDB"
        cp "${QDB_MANIFESTS_PATH}/manifest.yaml" "${TMP_MANIFESTS_FOLDER}/qdb_manifest.yaml"
        kubectl apply --namespace ${QDB_NAMESPACE} -f "${TMP_MANIFESTS_FOLDER}/qdb_manifest.yaml"

        echo ">>> Waiting QuestDB statefulset to be created..."
        while ! kubectl get --namespace ${QDB_NAMESPACE} statefulset/questdb &> /dev/null; do
            printf "%c" "."
            sleep 1
        done

        # Wait for statefulset condition "Available=True" does not work
        # Wait for statefulset condition "jsonpath='{.status.readyReplicas}'=3" throws error:
        #   "error: readyReplicas is not found"
        # Workaround: Check the pods are ready
        #echo ">>> QuestDB statefulset created. Waiting for readiness condition..."
        #kubectl wait --namespace  ${QDB_NAMESPACE} --for=condition=Available=True --timeout=300s statefulset/questdb
        #kubectl wait --namespace ${QDB_NAMESPACE} --for=jsonpath='{.status.readyReplicas}'=3 --timeout=300s \
        #    statefulset/questdb
        echo ">>> QuestDB statefulset created. Waiting QuestDB pods to be created..."
        while ! kubectl get --namespace ${QDB_NAMESPACE} pod/questdb-0 &> /dev/null; do
            printf "%c" "."
            sleep 1
        done
        kubectl wait --namespace ${QDB_NAMESPACE} --for=condition=Ready --timeout=300s pod/questdb-0
    fi
    echo

    echo "QuestDB Port Mapping"
    echo ">>> Expose QuestDB SQL port (8812->8812)"
    QDB_SQL_PORT=$(kubectl --namespace ${QDB_NAMESPACE} get service questdb-public -o 'jsonpath={.spec.ports[?(@.name=="sql")].port}')
    PATCH='{"data": {"'${QDB_SQL_PORT}'": "'${QDB_NAMESPACE}'/questdb-public:'${QDB_SQL_PORT}'"}}'
    kubectl patch configmap nginx-ingress-tcp-microk8s-conf --namespace ingress --patch "${PATCH}"

    PORT_MAP='{"containerPort": '${QDB_SQL_PORT}', "hostPort": '${QDB_SQL_PORT}'}'
    CONTAINER='{"name": "nginx-ingress-microk8s", "ports": ['${PORT_MAP}']}'
    PATCH='{"spec": {"template": {"spec": {"containers": ['${CONTAINER}']}}}}'
    kubectl patch daemonset nginx-ingress-microk8s-controller --namespace ingress --patch "${PATCH}"
    echo

    echo ">>> Expose QuestDB Influx Line Protocol port (9009->9009)"
    QDB_ILP_PORT=$(kubectl --namespace ${QDB_NAMESPACE} get service questdb-public -o 'jsonpath={.spec.ports[?(@.name=="ilp")].port}')
    PATCH='{"data": {"'${QDB_ILP_PORT}'": "'${QDB_NAMESPACE}'/questdb-public:'${QDB_ILP_PORT}'"}}'
    kubectl patch configmap nginx-ingress-tcp-microk8s-conf --namespace ingress --patch "${PATCH}"

    PORT_MAP='{"containerPort": '${QDB_ILP_PORT}', "hostPort": '${QDB_ILP_PORT}'}'
    CONTAINER='{"name": "nginx-ingress-microk8s", "ports": ['${PORT_MAP}']}'
    PATCH='{"spec": {"template": {"spec": {"containers": ['${CONTAINER}']}}}}'
    kubectl patch daemonset nginx-ingress-microk8s-controller --namespace ingress --patch "${PATCH}"
    echo

    echo ">>> Expose QuestDB HTTP Mgmt GUI port (9000->9000)"
    QDB_GUI_PORT=$(kubectl --namespace ${QDB_NAMESPACE} get service questdb-public -o 'jsonpath={.spec.ports[?(@.name=="http")].port}')
    PATCH='{"data": {"'${QDB_GUI_PORT}'": "'${QDB_NAMESPACE}'/questdb-public:'${QDB_GUI_PORT}'"}}'
    kubectl patch configmap nginx-ingress-tcp-microk8s-conf --namespace ingress --patch "${PATCH}"

    PORT_MAP='{"containerPort": '${QDB_GUI_PORT}', "hostPort": '${QDB_GUI_PORT}'}'
    CONTAINER='{"name": "nginx-ingress-microk8s", "ports": ['${PORT_MAP}']}'
    PATCH='{"spec": {"template": {"spec": {"containers": ['${CONTAINER}']}}}}'
    kubectl patch daemonset nginx-ingress-microk8s-controller --namespace ingress --patch "${PATCH}"
    echo
}

function qdb_undeploy() {
    echo "QuestDB"
    echo ">>> Checking if QuestDB is deployed..."
    if kubectl get --namespace ${QDB_NAMESPACE} statefulset/questdb &> /dev/null; then
        echo ">>> Undeploy QuestDB"
        kubectl delete --namespace ${QDB_NAMESPACE} -f "${TMP_MANIFESTS_FOLDER}/qdb_manifest.yaml" --ignore-not-found
    else
        echo ">>> QuestDB is not present; skipping step."
    fi
    echo

    echo "QuestDB Namespace"
    echo ">>> Delete QuestDB Namespace (if exists)"
    echo "NOTE: this step might take few minutes to complete!"
    kubectl delete namespace ${QDB_NAMESPACE} --ignore-not-found
    echo
}

# TODO: implement method to drop table
#function qdb_drop_table() {
#    echo "Drop table if exists"
#    QDB_CLIENT_URL="postgresql://${QDB_USERNAME}:${QDB_PASSWORD}@questdb-0:${QDB_SQL_PORT}/defaultdb?sslmode=require"
#    kubectl exec -it --namespace ${QDB_NAMESPACE} questdb-0 -- \
#        ./qdb sql --certs-dir=/qdb/qdb-certs --url=${QDB_CLIENT_URL} \
#        --execute "DROP TABLE IF EXISTS ${QDB_TABLE};"
#    echo
#}

if [ "$QDB_REDEPLOY" == "YES" ]; then
    qdb_undeploy
#elif [ "$QDB_DROP_TABLE_IF_EXISTS" == "YES" ]; then
#    qdb_drop_table
fi
qdb_deploy
