# Copyright 2022-2023 ETSI TeraFlowSDN - TFS OSG (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

import operator
from sqlalchemy import Column, DateTime, Enum, ForeignKey, String
from sqlalchemy.dialects.postgresql import UUID
from sqlalchemy.orm import relationship
from typing import Dict
from .enums.SliceStatus import ORM_SliceStatusEnum
from ._Base import _Base

class SliceModel(_Base):
    __tablename__ = 'slice'

    slice_uuid         = Column(UUID(as_uuid=False), primary_key=True)
    context_uuid       = Column(ForeignKey('context.context_uuid'), nullable=False)
    slice_name         = Column(String, nullable=True)
    slice_status       = Column(Enum(ORM_SliceStatusEnum), nullable=False)
    slice_owner_uuid   = Column(String, nullable=True)
    slice_owner_string = Column(String, nullable=True)
    created_at         = Column(DateTime, nullable=False)
    updated_at         = Column(DateTime, nullable=False)

    context         = relationship('ContextModel', back_populates='slices')
    slice_endpoints = relationship('SliceEndPointModel') # lazy='joined', back_populates='slice'
    slice_services  = relationship('SliceServiceModel') # lazy='joined', back_populates='slice'
    slice_subslices = relationship(
        'SliceSubSliceModel', primaryjoin='slice.c.slice_uuid == slice_subslice.c.slice_uuid')
    constraints     = relationship('ConstraintModel', passive_deletes=True) # lazy='joined', back_populates='slice'
    config_rules    = relationship('ConfigRuleModel', passive_deletes=True) # lazy='joined', back_populates='slice'

    def dump_id(self) -> Dict:
        return {
            'context_id': self.context.dump_id(),
            'slice_uuid': {'uuid': self.slice_uuid},
        }

    def dump(self) -> Dict:
        return {
            'slice_id'          : self.dump_id(),
            'name'              : self.slice_name,
            'slice_status'      : {'slice_status': self.slice_status.value},
            'slice_endpoint_ids': [
                slice_endpoint.endpoint.dump_id()
                for slice_endpoint in self.slice_endpoints
            ],
            'slice_constraints' : [
                constraint.dump()
                for constraint in sorted(self.constraints, key=operator.attrgetter('position'))
            ],
            'slice_config'      : {'config_rules': [
                config_rule.dump()
                for config_rule in sorted(self.config_rules, key=operator.attrgetter('position'))
            ]},
            'slice_service_ids': [
                slice_service.service.dump_id()
                for slice_service in self.slice_services
            ],
            'slice_subslice_ids': [
                slice_subslice.subslice.dump_id()
                for slice_subslice in self.slice_subslices
            ],
            'slice_owner': {
                'owner_uuid': {'uuid': self.slice_owner_uuid},
                'owner_string': self.slice_owner_string
            }
        }

class SliceEndPointModel(_Base):
    __tablename__ = 'slice_endpoint'

    slice_uuid    = Column(ForeignKey('slice.slice_uuid',       ondelete='CASCADE' ), primary_key=True)
    endpoint_uuid = Column(ForeignKey('endpoint.endpoint_uuid', ondelete='RESTRICT'), primary_key=True)

    slice    = relationship('SliceModel', back_populates='slice_endpoints', lazy='joined')
    endpoint = relationship('EndPointModel', lazy='joined') # back_populates='slice_endpoints'

class SliceServiceModel(_Base):
    __tablename__ = 'slice_service'

    slice_uuid   = Column(ForeignKey('slice.slice_uuid',     ondelete='CASCADE' ), primary_key=True)
    service_uuid = Column(ForeignKey('service.service_uuid', ondelete='RESTRICT'), primary_key=True)

    slice   = relationship('SliceModel', back_populates='slice_services', lazy='joined')
    service = relationship('ServiceModel', lazy='joined') # back_populates='slice_services'

class SliceSubSliceModel(_Base):
    __tablename__ = 'slice_subslice'

    slice_uuid    = Column(ForeignKey('slice.slice_uuid', ondelete='CASCADE' ), primary_key=True)
    subslice_uuid = Column(ForeignKey('slice.slice_uuid', ondelete='RESTRICT'), primary_key=True)

    slice    = relationship(
        'SliceModel', foreign_keys='SliceSubSliceModel.slice_uuid', back_populates='slice_subslices', lazy='joined')
    subslice = relationship('SliceModel', foreign_keys='SliceSubSliceModel.subslice_uuid', lazy='joined')
