import copy, logging, pytest
from common.Constants import DEFAULT_CONTEXT_UUID, DEFAULT_TOPOLOGY_UUID
from common.orm.Database import Database
from common.orm.Factory import get_database_backend, BackendEnum
from context.Config import GRPC_SERVICE_PORT, GRPC_MAX_WORKERS, GRPC_GRACE_PERIOD
from context.client.ContextClient import ContextClient
from context.proto.context_pb2 import (
    Context, ContextId, Device, DeviceId, DeviceOperationalStatusEnum, Empty, Link, LinkId, Service, ServiceId,
    ServiceStatusEnum, ServiceTypeEnum, Topology, TopologyId)
from context.service.grpc_server.ContextService import ContextService
from .example_objects import (
    CONTEXT, CONTEXT_ID, DEVICE1, DEVICE1_ID, DEVICE1_UUID, DEVICE2_ID, DEVICE2_UUID, LINK, LINK_ID, LINK_UUID,
    SERVICE, SERVICE_ID, SERVICE_UUID, TOPOLOGY, TOPOLOGY_ID)

LOGGER = logging.getLogger(__name__)
LOGGER.setLevel(logging.DEBUG)

GRPC_PORT = 10000 + GRPC_SERVICE_PORT # avoid privileged ports

SCENARIOS = [
    (BackendEnum.INMEMORY, {}),
    (BackendEnum.REDIS,    {
        'REDIS_SERVICE_HOST': '10.1.7.194',
        'REDIS_SERVICE_PORT': 30283,
        'REDIS_DATABASE_ID': 0,
    }),
]

@pytest.fixture(scope='session', ids=[str(scenario[0].value) for scenario in SCENARIOS], params=SCENARIOS)
def context_database(request):
    backend,settings = request.param
    LOGGER.info('Running fixture with backend={}, settings={}...'.format(str(backend), str(settings)))
    database_backend = get_database_backend(backend=backend, **settings)
    _database = Database(database_backend)
    return _database

@pytest.fixture(scope='session')
def context_service(context_database : Database): # pylint: disable=redefined-outer-name
    _service = ContextService(
        context_database, port=GRPC_PORT, max_workers=GRPC_MAX_WORKERS, grace_period=GRPC_GRACE_PERIOD)
    _service.start()
    yield _service
    _service.stop()

@pytest.fixture(scope='session')
def context_client(context_service : ContextService): # pylint: disable=redefined-outer-name
    _client = ContextClient(address='127.0.0.1', port=GRPC_PORT)
    yield _client
    _client.close()


def test_context_initial_state(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    context_database.clear_all()

    response = context_client.ListContextIds(Empty())
    assert len(response.context_ids) == 0

    response = context_client.ListContexts(Empty())
    assert len(response.contexts) == 0

    response = context_client.GetContext(ContextId(**CONTEXT_ID))
    assert len(response.context_id.context_uuid.uuid) == 0
    assert len(response.topology_ids) == 0
    assert len(response.service_ids) == 0

    db_entries = context_database.dump()
    LOGGER.info('----- Database Dump [{:3d} entries] -------------------------'.format(len(db_entries)))
    for db_entry in db_entries:
        LOGGER.info('  [{:>4s}] {:40s} :: {:s}'.format(*db_entry)) # pragma: no cover
    LOGGER.info('-----------------------------------------------------------')
    assert len(db_entries) == 0


def test_context_set(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    response = context_client.SetContext(Context(**CONTEXT))
    assert response.context_uuid.uuid == DEFAULT_CONTEXT_UUID

    db_entries = context_database.dump()
    LOGGER.info('----- Database Dump [{:3d} entries] -------------------------'.format(len(db_entries)))
    for db_entry in db_entries:
        LOGGER.info('  [{:>4s}] {:40s} :: {:s}'.format(*db_entry)) # pragma: no cover
    LOGGER.info('-----------------------------------------------------------')
    assert len(db_entries) == 2


def test_context_get_list(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    response = context_client.GetContext(ContextId(**CONTEXT_ID))
    assert response.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert len(response.topology_ids) == 0
    assert len(response.service_ids) == 0

    response = context_client.ListContextIds(Empty())
    assert len(response.context_ids) == 1
    assert response.context_ids[0].context_uuid.uuid == DEFAULT_CONTEXT_UUID

    response = context_client.ListContexts(Empty())
    assert len(response.contexts) == 1
    assert response.contexts[0].context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert len(response.contexts[0].topology_ids) == 0
    assert len(response.contexts[0].service_ids) == 0


def test_context_remove(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    context_client.RemoveContext(ContextId(**CONTEXT_ID))

    db_entries = context_database.dump()
    LOGGER.info('----- Database Dump [{:3d} entries] -------------------------'.format(len(db_entries)))
    for db_entry in db_entries:
        LOGGER.info('  [{:>4s}] {:40s} :: {:s}'.format(*db_entry)) # pragma: no cover
    LOGGER.info('-----------------------------------------------------------')
    assert len(db_entries) == 0


def test_topology_initial_state(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    context_database.clear_all()

    response = context_client.SetContext(Context(**CONTEXT))
    assert response.context_uuid.uuid == DEFAULT_CONTEXT_UUID

    response = context_client.ListTopologyIds(ContextId(**CONTEXT_ID))
    assert len(response.topology_ids) == 0

    response = context_client.ListTopologies(ContextId(**CONTEXT_ID))
    assert len(response.topologies) == 0

    response = context_client.GetTopology(TopologyId(**TOPOLOGY_ID))
    assert len(response.topology_id.topology_uuid.uuid) == 0
    assert len(response.topology_id.context_id.context_uuid.uuid) == 0
    assert len(response.device_ids) == 0
    assert len(response.link_ids) == 0

    db_entries = context_database.dump()
    LOGGER.info('----- Database Dump [{:3d} entries] -------------------------'.format(len(db_entries)))
    for db_entry in db_entries:
        LOGGER.info('  [{:>4s}] {:40s} :: {:s}'.format(*db_entry)) # pragma: no cover
    LOGGER.info('-----------------------------------------------------------')
    assert len(db_entries) == 2


def test_topology_set(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    response = context_client.SetTopology(Topology(**TOPOLOGY))
    assert response.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert response.topology_uuid.uuid == DEFAULT_TOPOLOGY_UUID

    db_entries = context_database.dump()
    LOGGER.info('----- Database Dump [{:3d} entries] -------------------------'.format(len(db_entries)))
    for db_entry in db_entries:
        LOGGER.info('  [{:>4s}] {:40s} :: {:s}'.format(*db_entry)) # pragma: no cover
    LOGGER.info('-----------------------------------------------------------')
    assert len(db_entries) == 5


def test_topology_get_list(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    response = context_client.GetTopology(TopologyId(**TOPOLOGY_ID))
    assert response.topology_id.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert response.topology_id.topology_uuid.uuid == DEFAULT_TOPOLOGY_UUID
    assert len(response.device_ids) == 0
    assert len(response.link_ids) == 0

    response = context_client.ListTopologyIds(ContextId(**CONTEXT_ID))
    assert len(response.topology_ids) == 1
    assert response.topology_ids[0].context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert response.topology_ids[0].topology_uuid.uuid == DEFAULT_TOPOLOGY_UUID

    response = context_client.ListTopologies(ContextId(**CONTEXT_ID))
    assert len(response.topologies) == 1
    assert response.topologies[0].topology_id.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert response.topologies[0].topology_id.topology_uuid.uuid == DEFAULT_TOPOLOGY_UUID
    assert len(response.topologies[0].device_ids) == 0
    assert len(response.topologies[0].link_ids) == 0


def test_topology_remove(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    context_client.RemoveTopology(TopologyId(**TOPOLOGY_ID))
    context_client.RemoveContext(ContextId(**CONTEXT_ID))

    db_entries = context_database.dump()
    LOGGER.info('----- Database Dump [{:3d} entries] -------------------------'.format(len(db_entries)))
    for db_entry in db_entries:
        LOGGER.info('  [{:>4s}] {:40s} :: {:s}'.format(*db_entry)) # pragma: no cover
    LOGGER.info('-----------------------------------------------------------')
    assert len(db_entries) == 0


def test_device_initial_state(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    context_database.clear_all()

    response = context_client.SetContext(Context(**CONTEXT))
    assert response.context_uuid.uuid == DEFAULT_CONTEXT_UUID

    response = context_client.SetTopology(Topology(**TOPOLOGY))
    assert response.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert response.topology_uuid.uuid == DEFAULT_TOPOLOGY_UUID

    response = context_client.ListDeviceIds(Empty())
    assert len(response.device_ids) == 0

    response = context_client.ListDevices(Empty())
    assert len(response.devices) == 0

    response = context_client.GetDevice(DeviceId(**DEVICE1_ID))
    assert len(response.device_id.device_uuid.uuid) == 0
    assert len(response.device_type) == 0
    assert len(response.device_config.config_rules) == 0
    assert response.device_operational_status == DeviceOperationalStatusEnum.DEVICEOPERATIONALSTATUS_UNDEFINED
    assert len(response.device_drivers) == 0
    assert len(response.device_endpoints) == 0

    db_entries = context_database.dump()
    LOGGER.info('----- Database Dump [{:3d} entries] -------------------------'.format(len(db_entries)))
    for db_entry in db_entries:
        LOGGER.info('  [{:>4s}] {:40s} :: {:s}'.format(*db_entry)) # pragma: no cover
    LOGGER.info('-----------------------------------------------------------')
    assert len(db_entries) == 5


def test_device_set(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    response = context_client.SetDevice(Device(**DEVICE1))
    assert response.device_uuid.uuid == DEVICE1_UUID

    db_entries = context_database.dump()
    LOGGER.info('----- Database Dump [{:3d} entries] -------------------------'.format(len(db_entries)))
    for db_entry in db_entries:
        LOGGER.info('  [{:>4s}] {:40s} :: {:s}'.format(*db_entry)) # pragma: no cover
    LOGGER.info('-----------------------------------------------------------')
    assert len(db_entries) == 25


def test_device_get_list(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    response = context_client.GetDevice(DeviceId(**DEVICE1_ID))
    assert response.device_id.device_uuid.uuid == DEVICE1_UUID
    assert response.device_type == 'packet-router'
    assert len(response.device_config.config_rules) == 3
    assert response.device_operational_status == DeviceOperationalStatusEnum.DEVICEOPERATIONALSTATUS_ENABLED
    assert len(response.device_drivers) == 2
    assert len(response.device_endpoints) == 3

    response = context_client.ListDeviceIds(Empty())
    assert len(response.device_ids) == 1
    assert response.device_ids[0].device_uuid.uuid == DEVICE1_UUID

    response = context_client.ListDevices(Empty())
    assert len(response.devices) == 1
    assert response.devices[0].device_id.device_uuid.uuid == DEVICE1_UUID
    assert response.devices[0].device_type == 'packet-router'
    assert len(response.devices[0].device_config.config_rules) == 3
    assert response.devices[0].device_operational_status == DeviceOperationalStatusEnum.DEVICEOPERATIONALSTATUS_ENABLED
    assert len(response.devices[0].device_drivers) == 2
    assert len(response.devices[0].device_endpoints) == 3


def test_device_add_to_topology(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    TOPOLOGY_WITH_DEVICE = copy.deepcopy(TOPOLOGY)
    TOPOLOGY_WITH_DEVICE['device_ids'].append(DEVICE1_ID)
    response = context_client.SetTopology(Topology(**TOPOLOGY_WITH_DEVICE))
    assert response.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert response.topology_uuid.uuid == DEFAULT_TOPOLOGY_UUID

    response = context_client.GetTopology(TopologyId(**TOPOLOGY_ID))
    assert response.topology_id.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert response.topology_id.topology_uuid.uuid == DEFAULT_TOPOLOGY_UUID
    assert len(response.device_ids) == 1
    assert response.device_ids[0].device_uuid.uuid == DEVICE1_UUID
    assert len(response.link_ids) == 0

    db_entries = context_database.dump()
    LOGGER.info('----- Database Dump [{:3d} entries] -------------------------'.format(len(db_entries)))
    for db_entry in db_entries:
        LOGGER.info('  [{:>4s}] {:40s} :: {:s}'.format(*db_entry)) # pragma: no cover
    LOGGER.info('-----------------------------------------------------------')
    assert len(db_entries) == 25


def test_device_remove(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    context_client.RemoveDevice(DeviceId(**DEVICE1_ID))
    context_client.RemoveTopology(TopologyId(**TOPOLOGY_ID))
    context_client.RemoveContext(ContextId(**CONTEXT_ID))

    db_entries = context_database.dump()
    LOGGER.info('----- Database Dump [{:3d} entries] -------------------------'.format(len(db_entries)))
    for db_entry in db_entries:
        LOGGER.info('  [{:>4s}] {:40s} :: {:s}'.format(*db_entry)) # pragma: no cover
    LOGGER.info('-----------------------------------------------------------')
    assert len(db_entries) == 0


def test_link_initial_state(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    context_database.clear_all()

    response = context_client.SetContext(Context(**CONTEXT))
    assert response.context_uuid.uuid == DEFAULT_CONTEXT_UUID

    response = context_client.SetTopology(Topology(**TOPOLOGY))
    assert response.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert response.topology_uuid.uuid == DEFAULT_TOPOLOGY_UUID

    response = context_client.ListLinkIds(Empty())
    assert len(response.link_ids) == 0

    response = context_client.ListLinks(Empty())
    assert len(response.links) == 0

    response = context_client.GetLink(LinkId(**LINK_ID))
    assert len(response.link_id.link_uuid.uuid) == 0
    assert len(response.link_endpoint_ids) == 0

    db_entries = context_database.dump()
    LOGGER.info('----- Database Dump [{:3d} entries] -------------------------'.format(len(db_entries)))
    for db_entry in db_entries:
        LOGGER.info('  [{:>4s}] {:40s} :: {:s}'.format(*db_entry)) # pragma: no cover
    LOGGER.info('-----------------------------------------------------------')
    assert len(db_entries) == 5


def test_link_set(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    response = context_client.SetLink(Link(**LINK))
    assert response.link_uuid.uuid == LINK_UUID

    db_entries = context_database.dump()
    LOGGER.info('----- Database Dump [{:3d} entries] -------------------------'.format(len(db_entries)))
    for db_entry in db_entries:
        LOGGER.info('  [{:>4s}] {:40s} :: {:s}'.format(*db_entry)) # pragma: no cover
    LOGGER.info('-----------------------------------------------------------')
    assert len(db_entries) == 30


def test_link_get_list(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    response = context_client.GetLink(LinkId(**LINK_ID))
    assert response.link_id.link_uuid.uuid == LINK_UUID
    assert len(response.link_endpoint_ids) == 2

    response = context_client.ListLinkIds(Empty())
    assert len(response.link_ids) == 1
    assert response.link_ids[0].link_uuid.uuid == LINK_UUID

    response = context_client.ListLinks(Empty())
    assert len(response.links) == 1
    assert response.links[0].link_id.link_uuid.uuid == LINK_UUID
    assert len(response.links[0].link_endpoint_ids) == 2


def test_link_add_to_topology(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    TOPOLOGY_WITH_LINK = copy.deepcopy(TOPOLOGY)
    TOPOLOGY_WITH_LINK['link_ids'].append(LINK_ID)
    response = context_client.SetTopology(Topology(**TOPOLOGY_WITH_LINK))
    assert response.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert response.topology_uuid.uuid == DEFAULT_TOPOLOGY_UUID

    response = context_client.GetTopology(TopologyId(**TOPOLOGY_ID))
    assert response.topology_id.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert response.topology_id.topology_uuid.uuid == DEFAULT_TOPOLOGY_UUID
    assert len(response.device_ids) == 2
    assert response.device_ids[0].device_uuid.uuid == DEVICE1_UUID
    assert response.device_ids[1].device_uuid.uuid == DEVICE2_UUID
    assert len(response.link_ids) == 1
    assert response.link_ids[0].link_uuid.uuid == LINK_UUID

    db_entries = context_database.dump()
    LOGGER.info('----- Database Dump [{:3d} entries] -------------------------'.format(len(db_entries)))
    for db_entry in db_entries:
        LOGGER.info('  [{:>4s}] {:40s} :: {:s}'.format(*db_entry)) # pragma: no cover
    LOGGER.info('-----------------------------------------------------------')
    assert len(db_entries) == 32


def test_link_remove(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    context_client.RemoveLink(LinkId(**LINK_ID))
    context_client.RemoveDevice(DeviceId(**DEVICE1_ID))
    context_client.RemoveDevice(DeviceId(**DEVICE2_ID))
    context_client.RemoveTopology(TopologyId(**TOPOLOGY_ID))
    context_client.RemoveContext(ContextId(**CONTEXT_ID))

    db_entries = context_database.dump()
    LOGGER.info('----- Database Dump [{:3d} entries] -------------------------'.format(len(db_entries)))
    for db_entry in db_entries:
        LOGGER.info('  [{:>4s}] {:40s} :: {:s}'.format(*db_entry)) # pragma: no cover
    LOGGER.info('-----------------------------------------------------------')
    assert len(db_entries) == 0


def test_service_initial_state(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    context_database.clear_all()

    response = context_client.SetContext(Context(**CONTEXT))
    assert response.context_uuid.uuid == DEFAULT_CONTEXT_UUID

    response = context_client.ListServiceIds(ContextId(**CONTEXT_ID))
    assert len(response.service_ids) == 0

    response = context_client.ListServices(ContextId(**CONTEXT_ID))
    assert len(response.services) == 0

    response = context_client.GetService(ServiceId(**SERVICE_ID))
    assert len(response.service_id.service_uuid.uuid) == 0
    assert len(response.service_id.context_id.context_uuid.uuid) == 0
    assert response.service_type == ServiceTypeEnum.SERVICETYPE_UNKNOWN
    assert len(response.service_endpoint_ids) == 0
    assert len(response.service_constraints) == 0
    assert response.service_status.service_status == ServiceStatusEnum.SERVICESTATUS_UNDEFINED
    assert len(response.service_config.config_rules) == 0

    db_entries = context_database.dump()
    LOGGER.info('----- Database Dump [{:3d} entries] -------------------------'.format(len(db_entries)))
    for db_entry in db_entries:
        LOGGER.info('  [{:>4s}] {:40s} :: {:s}'.format(*db_entry)) # pragma: no cover
    LOGGER.info('-----------------------------------------------------------')
    assert len(db_entries) == 2


def test_service_set(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    response = context_client.SetService(Service(**SERVICE))
    assert response.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert response.service_uuid.uuid == SERVICE_UUID

    db_entries = context_database.dump()
    LOGGER.info('----- Database Dump [{:3d} entries] -------------------------'.format(len(db_entries)))
    for db_entry in db_entries:
        LOGGER.info('  [{:>4s}] {:40s} :: {:s}'.format(*db_entry)) # pragma: no cover
    LOGGER.info('-----------------------------------------------------------')
    assert len(db_entries) == 42


def test_service_get_list(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    response = context_client.GetService(ServiceId(**SERVICE_ID))
    assert response.service_id.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert response.service_id.service_uuid.uuid == SERVICE_UUID
    assert response.service_type == ServiceTypeEnum.SERVICETYPE_L3NM
    assert len(response.service_endpoint_ids) == 2
    assert len(response.service_constraints) == 2
    assert response.service_status.service_status == ServiceStatusEnum.SERVICESTATUS_ACTIVE
    assert len(response.service_config.config_rules) == 3

    response = context_client.ListServiceIds(ContextId(**CONTEXT_ID))
    assert len(response.service_ids) == 1
    assert response.service_ids[0].context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert response.service_ids[0].service_uuid.uuid == SERVICE_UUID

    response = context_client.ListServices(ContextId(**CONTEXT_ID))
    assert len(response.services) == 1
    assert response.services[0].service_id.context_id.context_uuid.uuid == DEFAULT_CONTEXT_UUID
    assert response.services[0].service_id.service_uuid.uuid == SERVICE_UUID
    assert response.services[0].service_type == ServiceTypeEnum.SERVICETYPE_L3NM
    assert len(response.services[0].service_endpoint_ids) == 2
    assert len(response.services[0].service_constraints) == 2
    assert response.services[0].service_status.service_status == ServiceStatusEnum.SERVICESTATUS_ACTIVE
    assert len(response.services[0].service_config.config_rules) == 3


def test_service_remove(
    context_client : ContextClient, context_database : Database): # pylint: disable=redefined-outer-name

    context_client.RemoveService(ServiceId(**SERVICE_ID))
    context_client.RemoveDevice(DeviceId(**DEVICE1_ID))
    context_client.RemoveDevice(DeviceId(**DEVICE2_ID))
    context_client.RemoveTopology(TopologyId(**TOPOLOGY_ID))
    context_client.RemoveContext(ContextId(**CONTEXT_ID))

    db_entries = context_database.dump()
    LOGGER.info('----- Database Dump [{:3d} entries] -------------------------'.format(len(db_entries)))
    for db_entry in db_entries:
        LOGGER.info('  [{:>4s}] {:40s} :: {:s}'.format(*db_entry)) # pragma: no cover
    LOGGER.info('-----------------------------------------------------------')
    assert len(db_entries) == 0
