# Copyright 2022-2023 ETSI TeraFlowSDN - TFS OSG (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#
import grpc
from flask import current_app, redirect, render_template, Blueprint, flash, session, url_for
from common.proto.context_pb2 import ContextId, Slice, SliceId, SliceStatusEnum
from common.tools.context_queries.EndPoint import get_endpoint_names
from context.client.ContextClient import ContextClient
from slice.client.SliceClient import SliceClient

slice = Blueprint('slice', __name__, url_prefix='/slice')

context_client = ContextClient()
slice_client = SliceClient()

@slice.get('/')
def home():
    context_uuid = session.get('context_uuid', '-')
    if context_uuid == "-":
        flash("Please select a context!", "warning")
        return redirect(url_for("main.home"))
    request = ContextId()
    request.context_uuid.uuid = context_uuid
    context_client.connect()
    try:
        slice_list = context_client.ListSlices(request)
        slices = slice_list.slices
        context_found = True
    except grpc.RpcError as e:
        if e.code() != grpc.StatusCode.NOT_FOUND: raise
        if e.details() != 'Context({:s}) not found'.format(context_uuid): raise
        slices = []
        context_found = False

    if context_found:
        endpoint_ids = []
        for slice_ in slices:
            endpoint_ids.extend(slice_.slice_endpoint_ids)
        device_names, endpoints_data = get_endpoint_names(context_client, endpoint_ids)
    else:
        device_names, endpoints_data = [],[]

    context_client.close()

    return render_template(
        'slice/home.html', slices=slices, device_names=device_names, endpoints_data=endpoints_data,
        context_not_found=not context_found, sse=SliceStatusEnum)


@slice.route('add', methods=['GET', 'POST'])
def add():
    flash('Add slice route called', 'danger')
    raise NotImplementedError()
    return render_template('slice/home.html')


@slice.get('<path:slice_uuid>/detail')
def detail(slice_uuid: str):
    context_uuid = session.get('context_uuid', '-')
    if context_uuid == "-":
        flash("Please select a context!", "warning")
        return redirect(url_for("main.home"))
    
    request: SliceId = SliceId()
    request.slice_uuid.uuid = slice_uuid
    request.context_id.context_uuid.uuid = context_uuid
    req = ContextId()
    req.context_uuid.uuid = context_uuid
    try:
        context_client.connect()
        response: Slice = context_client.GetSlice(request)
        services = context_client.ListServices(req)

        endpoint_ids = []
        endpoint_ids.extend(response.slice_endpoint_ids)
        device_names, endpoints_data = get_endpoint_names(context_client, endpoint_ids)

        context_client.close()
    except Exception as e:
        flash('The system encountered an error and cannot show the details of this slice.', 'warning')
        current_app.logger.exception(e)
        return redirect(url_for('slice.home'))
    return render_template(
        'slice/detail.html', slice=response, device_names=device_names, endpoints_data=endpoints_data,
        sse=SliceStatusEnum, services=services)

#@slice.get('<path:slice_uuid>/delete')
#def delete(slice_uuid: str):
#    context_uuid = session.get('context_uuid', '-')
#    if context_uuid == "-":
#        flash("Please select a context!", "warning")
#        return redirect(url_for("main.home"))
#
#    try:
#        request = SliceId()
#        request.slice_uuid.uuid = slice_uuid
#        request.context_id.context_uuid.uuid = context_uuid
#        slice_client.connect()
#        response = slice_client.DeleteSlice(request)
#        slice_client.close()
#
#        flash('Slice "{:s}" deleted successfully!'.format(slice_uuid), 'success')
#    except Exception as e:
#        flash('Problem deleting slice "{:s}": {:s}'.format(slice_uuid, str(e.details())), 'danger')
#        current_app.logger.exception(e) 
#    return redirect(url_for('slice.home'))
