from __future__ import annotations
from typing import TYPE_CHECKING, Dict
from ..entity._Entity import _Entity
from ..entity.EntityAttributes import EntityAttributes
from .Keys import KEY_ENDPOINT

if TYPE_CHECKING:
    from .Context import Context
    from .Device import Device

VALIDATORS = {
    'port_type': lambda v: v is None or isinstance(v, str),
}

class Endpoint(_Entity):
    def __init__(self, endpoint_uuid : str, parent : 'Device'):
        super().__init__(parent=parent)
        self._endpoint_uuid = endpoint_uuid
        self._device_uuid = self._parent.device_uuid
        self._topology_uuid = self._parent.topology_uuid
        self._context_uuid = self._parent.context_uuid
        self._attributes = EntityAttributes(self, KEY_ENDPOINT, VALIDATORS)

    @property
    def parent(self) -> 'Device': return self._parent

    @property
    def context(self) -> 'Context': return self._parent.context

    @property
    def context_uuid(self) -> str: return self.context.context_uuid

    @property
    def topology_uuid(self) -> str: return self.parent.topology_uuid

    @property
    def device_uuid(self) -> str: return self.parent.device_uuid

    @property
    def endpoint_uuid(self) -> str: return self._endpoint_uuid

    @property
    def attributes(self) -> EntityAttributes: return self._attributes

    def create(self, port_type : str) -> 'Endpoint':
        self.update(update_attributes={
            'port_type': port_type
        })
        self.parent.endpoints.add(self._endpoint_uuid)
        return self

    def update(self, update_attributes={}, remove_attributes=[]) -> 'Endpoint':
        self.attributes.update(update_attributes=update_attributes, remove_attributes=remove_attributes)
        return self

    def delete(self) -> None:
        self.attributes.delete()
        self.parent.endpoints.delete(self._endpoint_uuid)

    def dump_uuid(self) -> Dict:
        return {
            'topoId': {'uuid': self.topology_uuid},
            'dev_id': {'uuid': self.device_uuid},
            'port_id': {'uuid': self.endpoint_uuid},
        }

    def dump(self) -> Dict:
        attributes = self.attributes.get()
        return {
            'port_id': self.dump_uuid(),
            'port_type': attributes.get('port_type', None),
        }
