package eu.teraflow.automation;

import static org.assertj.core.api.Assertions.assertThat;

import automation.Automation;
import automation.Automation.DeviceRole;
import automation.Automation.DeviceRoleId;
import automation.AutomationService;
import context.ContextOuterClass;
import eu.teraflow.automation.context.ContextGateway;
import eu.teraflow.automation.device.DeviceGateway;
import eu.teraflow.automation.device.model.*;
import eu.teraflow.automation.device.model.Device;
import io.quarkus.grpc.GrpcClient;
import io.quarkus.test.junit.QuarkusTest;
import io.quarkus.test.junit.mockito.InjectMock;
import io.smallrye.mutiny.Uni;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import javax.inject.Inject;
import org.jboss.logging.Logger;
import org.junit.jupiter.api.Test;
import org.mockito.Mockito;

@QuarkusTest
class AutomationServiceTest {
    private static final Logger LOGGER = Logger.getLogger(AutomationServiceTest.class);

    @GrpcClient AutomationService client;
    private final Serializer serializer;

    @InjectMock DeviceGateway deviceGateway;
    @InjectMock ContextGateway contextGateway;

    @Inject
    AutomationServiceTest(Serializer serializer) {
        this.serializer = serializer;
    }

    @Test
    void shouldAddDeviceRole() throws ExecutionException, InterruptedException, TimeoutException {
        final var message = new CompletableFuture<>();
        final var DEVICE_ID = "0f14d0ab-9608-7862-a9e4-5ed26688389b";
        final var DEVICE_ROLE_ID = "0f14d0ab-9608-7862-a9e4-5ed26688389a";
        final var DEVICE_TYPE = "ztp";

        final var emptyDeviceConfig = new DeviceConfig(List.of());
        final var disabledDevice =
                new Device(DEVICE_ID, DEVICE_TYPE, emptyDeviceConfig, DeviceOperationalStatus.DISABLED);
        Mockito.when(contextGateway.getDevice(Mockito.any()))
                .thenReturn(Uni.createFrom().item(disabledDevice));

        final var configRule = new ConfigRule(ConfigActionEnum.SET, "001", "initial-configuration");
        final var initialDeviceConfig = new DeviceConfig(List.of(configRule));
        Mockito.when(deviceGateway.getInitialConfiguration(Mockito.any()))
                .thenReturn(Uni.createFrom().item(initialDeviceConfig));

        Mockito.when(deviceGateway.configureDevice(Mockito.any()))
                .thenReturn(Uni.createFrom().item(DEVICE_ID));

        // TODO 1. add (de)serialization for device role (and its components) in serializer module
        // TODO 2. use it here
        final var serializedDeviceRoleId =
                DeviceRoleId.newBuilder()
                        .setDevRoleId(serializer.serializeUuid(DEVICE_ROLE_ID))
                        .setDevId(serializer.serializeDeviceId(DEVICE_ID))
                        .build();
        final var serializedDeviceRole =
                DeviceRole.newBuilder().setDevRoleId(serializedDeviceRoleId).build();
        client
                .ztpAdd(serializedDeviceRole)
                .subscribe()
                .with(
                        deviceRoleState -> {
                            LOGGER.infof("Received %s", deviceRoleState);
                            final var devRoleId = deviceRoleState.getDevRoleId();

                            final var deviceId = serializer.deserialize(devRoleId.getDevId());
                            assertThat(deviceId).isEqualTo(DEVICE_ID);

                            final var devRoleUuid = serializer.deserialize(devRoleId.getDevRoleId());
                            message.complete(devRoleUuid);
                        });
        assertThat(message.get(5, TimeUnit.SECONDS)).isEqualTo(DEVICE_ROLE_ID);
    }

    @Test
    void shouldUpdateDeviceRole() throws ExecutionException, InterruptedException, TimeoutException {
        CompletableFuture<String> message = new CompletableFuture<>();
        final var UUID_VALUE = "0f14d0ab-9605-7862-a9e4-5ed26688389b";

        final var uuid = serializer.serializeUuid(UUID_VALUE);
        final var deviceRoleId = DeviceRoleId.newBuilder().setDevRoleId(uuid).build();
        final var deviceRole = DeviceRole.newBuilder().setDevRoleId(deviceRoleId).build();

        client
                .ztpUpdate(deviceRole)
                .subscribe()
                .with(
                        deviceRoleState -> {
                            LOGGER.infof("Received response %s", deviceRoleState);
                            message.complete(deviceRoleState.getDevRoleId().toString());
                        });
        assertThat(message.get(5, TimeUnit.SECONDS)).contains(UUID_VALUE);
    }

    @Test
    void shouldGetDeviceRole() throws ExecutionException, InterruptedException, TimeoutException {
        CompletableFuture<String> message = new CompletableFuture<>();
        final var UUID_VALUE = "0f14d0ab-9605-7862-a9e4-5ed26688389b";

        final var uuid = serializer.serializeUuid(UUID_VALUE);
        final var deviceRoleId = DeviceRoleId.newBuilder().setDevRoleId(uuid).build();

        client
                .ztpGetDeviceRole(deviceRoleId)
                .subscribe()
                .with(
                        deviceRole -> {
                            LOGGER.infof("Received response %s", deviceRole);
                            message.complete(deviceRole.getDevRoleId().toString());
                        });
        assertThat(message.get(5, TimeUnit.SECONDS)).contains(UUID_VALUE);
    }

    @Test
    void shouldGetAllDeviceRolesByDeviceId()
            throws ExecutionException, InterruptedException, TimeoutException {
        CompletableFuture<String> message = new CompletableFuture<>();

        final var uuid = serializer.serializeUuid("0f14d0ab-9605-7862-a9e4-5ed26688389b");
        final var deviceId = ContextOuterClass.DeviceId.newBuilder().setDeviceUuid(uuid).build();

        client
                .ztpGetDeviceRolesByDeviceId(deviceId)
                .subscribe()
                .with(
                        deviceRoleList -> {
                            LOGGER.infof("Received response %s", deviceRoleList);
                            message.complete(deviceRoleList.toString());
                        });
        assertThat(message.get(5, TimeUnit.SECONDS)).isEmpty();
    }

    @Test
    void shouldDeleteDeviceRole() throws ExecutionException, InterruptedException, TimeoutException {
        CompletableFuture<String> message = new CompletableFuture<>();
        final var UUID_VALUE = "0f14d0ab-9605-7862-a9e4-5ed26688389b";

        final var uuid = serializer.serializeUuid(UUID_VALUE);
        final var deviceRoleId = DeviceRoleId.newBuilder().setDevRoleId(uuid).build();
        final var deviceRole = DeviceRole.newBuilder().setDevRoleId(deviceRoleId).build();

        client
                .ztpDelete(deviceRole)
                .subscribe()
                .with(
                        deviceRoleState -> {
                            LOGGER.infof("Received response %s", deviceRoleState);
                            message.complete(deviceRoleState.getDevRoleId().toString());
                        });
        assertThat(message.get(5, TimeUnit.SECONDS)).contains(UUID_VALUE);
    }

    @Test
    void shouldDeleteAllDevicesRolesByDeviceId()
            throws ExecutionException, InterruptedException, TimeoutException {
        CompletableFuture<String> message = new CompletableFuture<>();
        final var empty = Automation.Empty.newBuilder().build();

        client
                .ztpDeleteAll(empty)
                .subscribe()
                .with(
                        deletionResult -> {
                            LOGGER.infof("Received response %s", deletionResult);
                            message.complete(deletionResult.toString());
                        });
        assertThat(message.get(5, TimeUnit.SECONDS)).isEmpty();
    }
}
