/*
* Copyright 2022-2023 ETSI TeraFlowSDN - TFS OSG (https://tfs.etsi.org/)
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

package org.etsi.tfs.ztp;

import io.smallrye.mutiny.Uni;
import jakarta.enterprise.context.ApplicationScoped;
import jakarta.inject.Inject;
import org.etsi.tfs.ztp.context.ContextService;
import org.etsi.tfs.ztp.context.model.Device;
import org.etsi.tfs.ztp.context.model.DeviceConfig;
import org.etsi.tfs.ztp.device.DeviceService;
import org.jboss.logging.Logger;

@ApplicationScoped
public class ZtpServiceImpl implements ZtpService {
    private static final Logger LOGGER = Logger.getLogger(ZtpServiceImpl.class);
    // private static final String MESSAGE = "Retrieved %s";

    private final DeviceService deviceService;
    private final ContextService contextService;

    @Inject
    public ZtpServiceImpl(DeviceService deviceService, ContextService contextService) {
        this.deviceService = deviceService;
        this.contextService = contextService;
    }

    @Override
    public Uni<Device> addDevice(String deviceId) {
        final var deserializedDeviceUni = contextService.getDevice(deviceId);

        deserializedDeviceUni
                .onFailure()
                .recoverWithNull()
                .subscribe()
                .with(
                        device -> {
                            final var id = deviceId;

                            if (device == null) {
                                LOGGER.warnf("%s is null. Ignoring...", device);
                                return;
                            }

                            if (device.isEnabled()) {
                                LOGGER.warnf("%s has already been enabled. Ignoring...", device);
                                return;
                            }

                            // LOGGER.infof(MESSAGE, device);

                            final var initialConfiguration =
                                    deviceService.getInitialConfiguration(device.getDeviceId());

                            device.enableDevice();
                            LOGGER.infof("Enabled device [%s]", id);

                            initialConfiguration
                                    .subscribe()
                                    .with(
                                            deviceConfig -> {
                                                device.setDeviceConfiguration(deviceConfig);
                                                final var configuredDeviceIdUni = deviceService.configureDevice(device);

                                                configuredDeviceIdUni
                                                        .subscribe()
                                                        .with(
                                                                configuredDeviceId ->
                                                                        LOGGER.infof(
                                                                                "Device [%s] has been successfully enabled and configured with %s.\n",
                                                                                id, deviceConfig));
                                            });
                        });

        return deserializedDeviceUni;
    }

    @Override
    public Uni<Device> deleteDevice(String deviceId) {
        final var deserializedDeviceUni = contextService.getDevice(deviceId);

        deserializedDeviceUni
                .onFailure()
                .recoverWithNull()
                .subscribe()
                .with(
                        device -> {
                            final var id = deviceId;

                            if (device == null) {
                                LOGGER.warnf("%s is null. Ignoring...", device);
                                return;
                            }

                            if (device.isDisabled()) {
                                LOGGER.warnf("%s has already been disabled. Ignoring...", device);
                                return;
                            }

                            device.disableDevice();
                            LOGGER.infof("Disabled device [%s]", id);

                            // LOGGER.infof(MESSAGE, device);

                            final var empty = deviceService.deleteDevice(device.getDeviceId());

                            empty
                                    .subscribe()
                                    .with(
                                            emptyMessage ->
                                                    LOGGER.infof("Device [%s] has been successfully deleted.\n", id));
                        });

        return deserializedDeviceUni;
    }

    @Override
    public Uni<Device> updateDevice(String deviceId, DeviceConfig deviceConfig) {
        final var deserializedDeviceUni = contextService.getDevice(deviceId);

        deserializedDeviceUni
                .onFailure()
                .recoverWithNull()
                .subscribe()
                .with(
                        device -> {
                            final var id = deviceId;

                            if (device == null) {
                                LOGGER.warnf("%s is null. Ignoring...", device);
                                return;
                            }

                            if (!device.isEnabled()) {
                                LOGGER.warnf("Cannot update disabled device %s. Ignoring...", device);
                                return;
                            }

                            // LOGGER.infof(MESSAGE, device);
                            device.setDeviceConfiguration(deviceConfig);
                            final var updatedDeviceIdUni = deviceService.configureDevice(device);

                            updatedDeviceIdUni
                                    .subscribe()
                                    .with(
                                            configuredDeviceId ->
                                                    LOGGER.infof(
                                                            "Device [%s] has been successfully updated with %s.\n",
                                                            id, deviceConfig));
                        });

        return deserializedDeviceUni;
    }
}
