/*
* Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
*
* Licensed under the Apache License, Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
*      http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law or agreed to in writing, software
* distributed under the License is distributed on an "AS IS" BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the specific language governing permissions and
* limitations under the License.
*/

package eu.teraflow.policy;

import context.ContextOuterClass.ServiceId;
import io.quarkus.grpc.GrpcService;
import io.smallrye.mutiny.Uni;
import javax.inject.Inject;
import policy.Policy;
import policy.Policy.PolicyRuleBasic;
import policy.Policy.PolicyRuleDevice;
import policy.Policy.PolicyRuleId;
import policy.Policy.PolicyRuleService;
import policy.Policy.PolicyRuleServiceList;
import policy.Policy.PolicyRuleState;

@GrpcService
public class PolicyGatewayImpl implements PolicyGateway {

    private final PolicyService policyService;
    private final Serializer serializer;

    @Inject
    public PolicyGatewayImpl(PolicyService policyService, Serializer serializer) {
        this.policyService = policyService;
        this.serializer = serializer;
    }

    @Override
    public Uni<PolicyRuleState> policyAddService(PolicyRuleService request) {
        final var policyRuleService = serializer.deserialize(request);

        return policyService
                .addPolicyService(policyRuleService)
                .onItem()
                .transform(serializer::serialize);
    }

    @Override
    public Uni<PolicyRuleState> policyUpdateService(PolicyRuleService request) {
        final var policyRuleService = serializer.deserialize(request);

        return policyService
                .updatePolicyService(policyRuleService)
                .onItem()
                .transform(serializer::serialize);
    }

    @Override
    public Uni<PolicyRuleState> policyAddDevice(PolicyRuleDevice request) {
        final var policyRuleDevice = serializer.deserialize(request);

        return policyService
                .addPolicyDevice(policyRuleDevice)
                .onItem()
                .transform(serializer::serialize);
    }

    @Override
    public Uni<PolicyRuleState> policyUpdateDevice(PolicyRuleDevice request) {
        final var policyRuleDevice = serializer.deserialize(request);

        return policyService
                .updatePolicyDevice(policyRuleDevice)
                .onItem()
                .transform(serializer::serialize);
    }

    @Override
    public Uni<PolicyRuleState> policyDelete(PolicyRuleId request) {
        final var policyRuleId = serializer.deserialize(request);

        return policyService.deletePolicy(policyRuleId).onItem().transform(serializer::serialize);
    }

    @Override
    public Uni<PolicyRuleService> getPolicyService(PolicyRuleId request) {
        final var policyRuleBasic = PolicyRuleBasic.newBuilder().setPolicyRuleId(request).build();

        return Uni.createFrom()
                .item(() -> PolicyRuleService.newBuilder().setPolicyRuleBasic(policyRuleBasic).build());
    }

    @Override
    public Uni<PolicyRuleDevice> getPolicyDevice(PolicyRuleId request) {
        final var policyRuleBasic = PolicyRuleBasic.newBuilder().setPolicyRuleId(request).build();

        return Uni.createFrom()
                .item(() -> PolicyRuleDevice.newBuilder().setPolicyRuleBasic(policyRuleBasic).build());
    }

    @Override
    public Uni<PolicyRuleServiceList> getPolicyByServiceId(ServiceId request) {
        return Uni.createFrom().item(() -> Policy.PolicyRuleServiceList.newBuilder().build());
    }
}
