# Copyright 2021-2023 H2020 TeraFlow (https://www.teraflow-h2020.eu/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

# RFC 8466 - L2VPN Service Model (L2SM)
# Ref: https://datatracker.ietf.org/doc/html/rfc8466

from compute.service.rest_server.RestServer import RestServer
from .Resources import (
    Connection, ConnectionIds, Connections, Context, ContextIds, Contexts, Device, DeviceIds, Devices, Link, LinkIds,
    Links, PolicyRule, PolicyRuleIds, PolicyRules, Service, ServiceIds, Services, Slice, SliceIds, Slices, Topologies,
    Topology, TopologyIds)

URL_PREFIX = '/api'

# Use 'path' type in Service and Sink because service_uuid and link_uuid might contain char '/' and Flask is unable to
# recognize them in 'string' type.
RESOURCES = [
    # (endpoint_name, resource_class, resource_url)
    ('api.context_ids',    ContextIds,    '/context_ids'),
    ('api.contexts',       Contexts,      '/contexts'),
    ('api.context',        Context,       '/context/<string:context_uuid>'),

    ('api.topology_ids',   TopologyIds,   '/context/<string:context_uuid>/topology_ids'),
    ('api.topologies',     Topologies,    '/context/<string:context_uuid>/topologies'),
    ('api.topology',       Topology,      '/context/<string:context_uuid>/topology/<string:topology_uuid>'),

    ('api.service_ids',    ServiceIds,    '/context/<string:context_uuid>/service_ids'),
    ('api.services',       Services,      '/context/<string:context_uuid>/services'),
    ('api.service',        Service,       '/context/<string:context_uuid>/service/<path:service_uuid>'),

    ('api.slice_ids',      SliceIds,      '/context/<string:context_uuid>/slice_ids'),
    ('api.slices',         Slices,        '/context/<string:context_uuid>/slices'),
    ('api.slice',          Slice,         '/context/<string:context_uuid>/slice/<path:slice_uuid>'),

    ('api.device_ids',     DeviceIds,     '/device_ids'),
    ('api.devices',        Devices,       '/devices'),
    ('api.device',         Device,        '/device/<string:device_uuid>'),

    ('api.link_ids',       LinkIds,       '/link_ids'),
    ('api.links',          Links,         '/links'),
    ('api.link',           Link,          '/link/<path:link_uuid>'),

    ('api.connection_ids', ConnectionIds, '/context/<string:context_uuid>/service/<path:service_uuid>/connection_ids'),
    ('api.connections',    Connections,   '/context/<string:context_uuid>/service/<path:service_uuid>/connections'),
    ('api.connection',     Connection,    '/connection/<path:connection_uuid>'),

    ('api.policyrule_ids', PolicyRuleIds, '/policyrule_ids'),
    ('api.policyrules',    PolicyRules,   '/policyrules'),
    ('api.policyrule',     PolicyRule,    '/policyrule/<string:policyrule_uuid>'),
]

def register_debug_api(rest_server : RestServer):
    for endpoint_name, resource_class, resource_url in RESOURCES:
        rest_server.add_resource(resource_class, URL_PREFIX + resource_url, endpoint=endpoint_name)
