# Copyright 2022-2024 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


import logging, threading, time
from flask import request
from flask_socketio import Namespace, join_room, leave_room
from nbi.service.NbiApplication import NbiApplication

LOGGER = logging.getLogger(__name__)

NAMESPACE_NAME = 'heartbeat'
NAMESPACE_URL  = '/heartbeat'

# WebSocket Heartbeat Namespace for debugging purposes
class HeartbeatHandler(Namespace):
    def on_connect(self):
        LOGGER.debug('Client {:s} connected'.format(str(request.sid)))
        join_room(NAMESPACE_NAME)

    def on_disconnect(self, reason):
        LOGGER.debug('Client {:s} disconnected: reason={:s}'.format(
            str(request.sid), str(reason)
        ))
        leave_room(NAMESPACE_NAME)

class HeartbeatThread(threading.Thread):
    INTERVAL = 1 # second

    def __init__(self, nbi_app : NbiApplication):
        super().__init__(daemon=True)
        self.nbi_app = nbi_app
        self.nbi_app.heartbeat_thread = self

    def run(self):
        interval   = HeartbeatThread.INTERVAL
        start_time = time.time()
        while True:
            time.sleep(interval)
            uptime = time.time() - start_time
            LOGGER.warning('emitting...')
            self.nbi_app.websocket_emit_message(
                'uptime', {'uptime_seconds': uptime},
                namespace=NAMESPACE_URL, to=NAMESPACE_NAME
            )

NAMESPACE_DESCRIPTOR = (NAMESPACE_NAME, HeartbeatHandler, NAMESPACE_URL)
