# Copyright 2022-2024 ETSI SDG TeraFlowSDN (TFS) (https://tfs.etsi.org/)
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.


import logging, time
from typing import Any, Optional
from flask import Flask, request
from flask_restful import Api, Resource
from flask_socketio import Namespace, SocketIO


LOGGER = logging.getLogger(__name__)

def log_request(response):
    timestamp = time.strftime('[%Y-%b-%d %H:%M]')
    LOGGER.info(
        '%s %s %s %s %s', timestamp, request.remote_addr, request.method,
        request.full_path, response.status
    )
    return response

class NbiApplication:
    def __init__(self, base_url : Optional[str] = None) -> None:
        if base_url is None: base_url = ''
        self.base_url = base_url

        self._app = Flask(__name__)
        self._app.config['SECRET_KEY'] = 'secret!'
        self._app.after_request(log_request)
        self._api = Api(self._app, prefix=base_url)
        #websocket_path = '/'.join([base_url.rstrip('/'), 'websocket'])
        #self._sio = SocketIO(self._app, path=base_url, cors_allowed_origins="*", logger=True, engineio_logger=True)
        self._sio = SocketIO(self._app, cors_allowed_origins="*", logger=True, engineio_logger=True)

        @self._sio.on_error_default  # handles all namespaces without an explicit error handler
        def default_error_handler(e):
            LOGGER.error('[default_error_handler] e={:s}'.format(str(e)))

    def add_rest_api_resource(self, resource_class : Resource, *urls, **kwargs) -> None:
        self._api.add_resource(resource_class, *urls, **kwargs)

    def add_websocket_namespace(self, namespace_class : Namespace, namespace_url : str) -> None:
        LOGGER.warning('[add_websocket_namespace] (before) self._sio.server={:s}'.format(str(self._sio.server)))
        LOGGER.warning('[add_websocket_namespace] (before) self._sio.server.namespace_handlers={:s}'.format(str(self._sio.server.namespace_handlers)))
        LOGGER.warning('[add_websocket_namespace] (before) self._sio.namespace_handlers={:s}'.format(str(self._sio.namespace_handlers)))
        self._sio.on_namespace(namespace_class(namespace_url))
        LOGGER.warning('[add_websocket_namespace] (after) self._sio.server={:s}'.format(str(self._sio.server)))
        LOGGER.warning('[add_websocket_namespace] (after) self._sio.server.namespace_handlers={:s}'.format(str(self._sio.server.namespace_handlers)))
        LOGGER.warning('[add_websocket_namespace] (after) self._sio.namespace_handlers={:s}'.format(str(self._sio.namespace_handlers)))

    def websocket_emit_message(
        self, event : str, *args : Any, namespace : str = "/", to : Optional[str] = None
    ) -> None:
        self._sio.emit(event, *args, namespace=namespace, to=to)

    def get_flask_app(self) -> Flask:
        return self._app

    def dump_configuration(self) -> None:
        LOGGER.debug('Configured REST-API Resources:')
        for resource in self._api.resources:
            LOGGER.debug(' - {:s}'.format(str(resource)))

        LOGGER.debug('Configured Flask Rules:')
        for rule in self._app.url_map.iter_rules():
            LOGGER.debug(' - {:s}'.format(str(rule)))

        LOGGER.debug('Configured WebSocket Namespaces:')
        for namespace in self._sio.server.handlers.keys():
            LOGGER.debug(' (server) - {:s}'.format(str(namespace)))

        for namespace in self._sio.namespace_handlers:
            LOGGER.debug(' (ns_hdls) - {:s}'.format(str(namespace)))

    def run_standalone(
        self, bind_address : Optional[str] = None, bind_port : Optional[int] = None,
        debug : bool = False, use_reloader : bool = False
    ) -> None:
        # Run method used when started in a standalone mode, i.e., outside gunicorn or
        # similar WSGI HTTP servers. Otherwise, use mechanism defined by the used
        # WSGI HTTP server.

        #logging.getLogger('werkzeug').setLevel(logging.WARNING)

        endpoint = 'http://{:s}:{:s}'.format(str(bind_address), str(bind_port))
        if self.base_url is not None:
            endpoint = '/'.join([endpoint.rstrip('/'), self.base_url])

        LOGGER.info('Listening on {:s}...'.format(endpoint))
        self._sio.run(
            self._app, host=bind_address, port=bind_port,
            debug=debug, use_reloader=use_reloader
        )

    def start_test_thread(
        self, bind_address : Optional[str] = None, bind_port : Optional[int] = None,
        debug : bool = False, use_reloader : bool = False
    ) -> None:
        # NOTE: To be used for testing purposes with pytest
        # Stop the thread through nbi_app.stop_test_thread()
        self._thread = self._sio.start_background_task(
            self._sio.run, self._app, host=bind_address, port=bind_port,
            debug=debug, use_reloader=use_reloader
        )

    def stop_test_thread(self):
        # NOTE: To be used for testing purposes with pytest
        # Start the thread through nbi_app.start_test_thread(...)
        if self._thread is None: return
        self._thread.join()
